<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Inertia\Inertia;
use Illuminate\Support\Str;

class GenericModelController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request, string $model)
    {
        $modelClass = $this->getModelClass($model);
        if (!$modelClass) {
            abort(404, 'Model not found.');
        }

        $query = $modelClass::query();
        $modelInstance = new $modelClass();
        $fillable = $modelInstance->getFillable();

        // Add search functionality
        if ($request->has('search') && $request->search) {
            $query->where(function ($q) use ($request, $fillable) {
                foreach ($fillable as $field) {
                    $q->orWhere($field, 'like', '%' . $request->search . '%');
                }
            });
        }

        // Add sorting
        $sortBy = $request->get('sort_by', 'created_at');
        $sortDirection = $request->get('sort_direction', 'desc');
        $query->orderBy($sortBy, $sortDirection);

        // Paginate results
        $items = $query->paginate(
            perPage: $request->get('per_page', 10),
            page: $request->get('page', 1)
        )->withQueryString();

        return Inertia::render('GenericModelPage', [
            'model' => Str::singular(Str::studly($model)),
            'items' => $items,
            'filters' => [
                'search' => $request->search,
                'sort_by' => $sortBy,
                'sort_direction' => $sortDirection,
                'per_page' => $request->get('per_page', 10)
            ]
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request, string $model)
    {
        $modelClass = $this->getModelClass($model);
        if (!$modelClass) {
            abort(404, 'Model not found.');
        }

        $modelInstance = new $modelClass();
        $fillable = $modelInstance->getFillable();

        // Dynamically get validation rules from a method on the model
        $validated = $request->validate($modelClass::getValidationRules());

        $modelClass::create($validated);

        return redirect()->route('generic.index', ['model' => Str::plural($model)])
            ->with('success', Str::singular($model) . ' created successfully.');
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $model, string $id)
    {
        $modelClass = $this->getModelClass($model);
        if (!$modelClass) {
            abort(404, 'Model not found.');
        }

        $item = $modelClass::findOrFail($id);

        $validated = $request->validate($modelClass::getValidationRules($item->id));

        $item->update($validated);

        return redirect()->route('generic.index', ['model' => Str::plural($model)])
            ->with('success', Str::singular($model) . ' updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $model, string $id)
    {
        $modelClass = $this->getModelClass($model);
        if (!$modelClass) {
            abort(404, 'Model not found.');
        }

        $item = $modelClass::findOrFail($id);
        $item->delete();

        return redirect()->route('generic.index', ['model' => Str::plural($model)])
            ->with('success', Str::singular($model) . ' deleted successfully.');
    }

    /**
     * Helper to resolve the model class.
     */
    protected function getModelClass(string $model)
    {
        $modelClass = 'App\\Models\\' . Str::studly(Str::singular($model));
        if (class_exists($modelClass)) {
            return $modelClass;
        }
        return null;
    }
}
