<?php

namespace App\Http\Controllers;

use App\Models\Payment;
use App\Models\Order;
use App\Models\Client;
use App\Models\Product;
use App\Models\Producer;
use App\Models\Batch;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;

class DashboardController extends Controller
{
    public function index()
    {
        // Existing revenue metrics
        $currentRevenue = Payment::whereMonth('payment_date', now()->month)
            ->whereYear('payment_date', now()->year)
            ->where('payment_status', 'paid')
            ->sum('amount');

        $previousRevenue = Payment::whereMonth('payment_date', now()->subMonth()->month)
            ->whereYear('payment_date', now()->subMonth()->year)
            ->where('payment_status', 'paid')
            ->sum('amount');

        $monthlyRevenue = Payment::select(
            DB::raw("to_char(payment_date, 'YYYY-MM') as month"),
            DB::raw('SUM(amount) as revenue')
        )
            ->where('payment_status', 'paid')
            ->groupBy('month')
            ->orderBy('month')
            ->get()
            ->map(fn($row) => [
                'month' => $row->month,
                'revenue' => (float) $row->revenue,
            ]);

        // Updated Reports - No Invoice model references

        // 1. Top Buyers - Calculate from orders and order_batches
        $topBuyers = Client::select(
            'clients.id',
            'clients.name as company_name',
            DB::raw('COUNT(orders.id) as total_orders'),
            DB::raw('COALESCE(SUM(order_batches.total_price), 0) as total_spent')
        )
            ->leftJoin('orders', 'clients.id', '=', 'orders.buyer_id')
            ->leftJoin('order_batches', 'orders.id', '=', 'order_batches.order_id')
            ->groupBy('clients.id', 'clients.name')
            ->orderBy('total_spent', 'DESC')
            ->orderBy('total_orders', 'DESC')
            ->limit(5)
            ->get();

        // 2. Pending Orders - Calculate from orders and order_batches
        $pendingOrders = Client::select(
            'clients.id',
            'clients.name as company_name',
            DB::raw('COUNT(orders.id) as pending_orders'),
            DB::raw('COALESCE(SUM(order_batches.total_price), 0) as total_pending_value')
        )
            ->leftJoin('orders', 'clients.id', '=', 'orders.buyer_id')
            ->leftJoin('order_batches', 'orders.id', '=', 'order_batches.order_id')
            ->where('orders.status', 'pending')
            ->groupBy('clients.id', 'clients.name')
            ->orderBy('total_pending_value', 'DESC')
            ->limit(5)
            ->get();

        // 3. Orders by Country - Calculate from orders and order_batches
        $ordersByCountry = Client::select(
            'clients.country',
            DB::raw('COUNT(orders.id) as total_orders'),
            DB::raw('COALESCE(SUM(order_batches.total_price), 0) as total_revenue')
        )
            ->leftJoin('orders', 'clients.id', '=', 'orders.buyer_id')
            ->leftJoin('order_batches', 'orders.id', '=', 'order_batches.order_id')
            ->groupBy('clients.country')
            ->orderBy('total_revenue', 'DESC')
            ->get();

        // 4. Best Selling Products - using order_batches
        $bestSellingProducts = Product::select(
            'products.id',
            'products.name as product_name',
            'producers.name as producer_name',
            DB::raw('COUNT(DISTINCT order_batches.order_id) as times_ordered'),
            DB::raw('COALESCE(SUM(order_batches.total_price), 0) as total_revenue')
        )
            ->join('producers', 'products.producer_id', '=', 'producers.id')
            ->leftJoin('batches', 'products.id', '=', 'batches.product_id')
            ->leftJoin('order_batches', 'batches.id', '=', 'order_batches.batch_id')
            ->leftJoin('orders', 'order_batches.order_id', '=', 'orders.id')
            ->groupBy('products.id', 'products.name', 'producers.name')
            ->orderBy('total_revenue', 'DESC')
            ->orderBy('times_ordered', 'DESC')
            ->limit(5)
            ->get();

        // 5. Revenue by Producer - using order_batches
        $revenueByProducer = Producer::select(
            'producers.id',
            'producers.name as company_name',
            DB::raw('COALESCE(SUM(order_batches.total_price), 0) as total_revenue')
        )
            ->join('products', 'producers.id', '=', 'products.producer_id')
            ->join('batches', 'products.id', '=', 'batches.product_id')
            ->join('order_batches', 'batches.id', '=', 'order_batches.batch_id')
            ->groupBy('producers.id', 'producers.name')
            ->orderBy('total_revenue', 'DESC')
            ->limit(5)
            ->get();

        // 6. Unpaid Orders (replaced Unpaid Invoices)
        $unpaidOrders = Order::select(
            'orders.id as order_key',
            'orders.date as order_date',
            'orders.status as order_status',
            'payments.due_date as payment_due_date',
            'payments.payment_status',
            'clients.id as buyer_id',
            'clients.name as buyer_name',
            'clients.billing_address'
        )
            ->join('clients', 'orders.buyer_id', '=', 'clients.id')
            ->leftJoin('payments', 'orders.id', '=', 'payments.order_id')
            ->where(function($query) {
                $query->where('payments.payment_status', '!=', 'paid')
                    ->orWhereNull('payments.payment_status');
            })
            ->orderBy('payments.due_date', 'DESC')
            ->limit(5)
            ->get();

        // 7. Payment Status Overview
        $paymentStatusOverview = Payment::select(
            'payment_status',
            DB::raw('COUNT(*) as count_payments'),
            DB::raw('SUM(amount) as total_amount'),
            DB::raw('ROUND((COUNT(*) * 100.0 / (SELECT COUNT(*) FROM payments)), 2) as percentage_of_payments')
        )
            ->groupBy('payment_status')
            ->orderBy('count_payments', 'DESC')
            ->get();

        // 8. Products with Poor Performance
        $poorPerformingProducts = Product::select(
            'products.id',
            'products.name as product_name',
            'producers.name as producer_name',
            DB::raw('COUNT(DISTINCT order_batches.order_id) as order_count')
        )
            ->join('producers', 'products.producer_id', '=', 'producers.id')
            ->leftJoin('batches', 'products.id', '=', 'batches.product_id')
            ->leftJoin('order_batches', 'batches.id', '=', 'order_batches.batch_id')
            ->leftJoin('orders', 'order_batches.order_id', '=', 'orders.id')
            ->where(function($query) {
                $query->where('orders.date', '>=', now()->subMonths(6))
                    ->orWhereNull('orders.date');
            })
            ->groupBy('products.id', 'products.name', 'producers.name')
            ->having(DB::raw('COUNT(DISTINCT order_batches.order_id)'), '<', 5)
            ->orderBy('order_count', 'ASC')
            ->limit(5)
            ->get();

        return Inertia::render('Dashboard', [
            'currentRevenue' => (float) $currentRevenue,
            'previousRevenue' => (float) $previousRevenue,
            'monthlyRevenue' => $monthlyRevenue,
            'topBuyers' => $topBuyers,
            'pendingOrders' => $pendingOrders,
            'ordersByCountry' => $ordersByCountry,
            'bestSellingProducts' => $bestSellingProducts,
            'revenueByProducer' => $revenueByProducer,
            'unpaidOrders' => $unpaidOrders, // Changed from unpaidInvoices
            'paymentStatusOverview' => $paymentStatusOverview,
            'poorPerformingProducts' => $poorPerformingProducts,
        ]);
    }
}
