<?php
namespace Database\Seeders;

use App\Models\Order;
use App\Models\Client;
use App\Models\Batch;
use App\Models\Transport;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use Illuminate\Support\Str;

class OrderSeeder extends Seeder
{
    private $statuses = ['pending', 'confirmed', 'processing', 'shipped', 'delivered', 'cancelled'];

    public function run(): void
    {
        $clients = Client::all();
        $transports = Transport::all();
        $batches = Batch::all();

        if ($transports->isEmpty()) {
            throw new \Exception("No transports found. Make sure TransportSeeder runs first.");
        }

        for ($i = 0; $i < 50; $i++) {
            $orderDate = Carbon::now()->subDays(rand(1, 365));
            $estimatedDelivery = $orderDate->copy()->addDays(rand(7, 30));
            $status = $this->statuses[array_rand($this->statuses)];

            $orderId = (string) Str::uuid();

            // Create order using DB facade to avoid any model issues
            DB::table('orders')->insert([
                'id' => $orderId,
                'date' => $orderDate->format('Y-m-d'),
                'status' => $status,
                'estimated_delivery_date' => $estimatedDelivery->format('Y-m-d'),
                'buyer_id' => $clients->random()->id,
                'receiver_id' => $clients->random()->id,
                'transport_id' => $transports->random()->id,
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            // Attach 1-5 batches to each order
            $orderBatches = $batches->random(rand(1, 5));

            foreach ($orderBatches as $batch) {
                $quantity = rand(10, 100);
                $pricePerUnit = $batch->product->price * (rand(80, 120) / 100);
                $totalPrice = $quantity * $pricePerUnit;

                // Cap the total price to prevent overflow - use lower limits
                $cappedTotalPrice = min($totalPrice, 50000.00); // Lower cap for individual batch totals
                $cappedPricePerUnit = min($pricePerUnit, 1000.00); // Lower cap for unit price

                DB::table('order_batches')->insert([
                    'id' => (string) Str::uuid(),
                    'order_id' => $orderId,
                    'batch_id' => $batch->id,
                    'quantity' => $quantity,
                    'price_per_unit' => (float) sprintf('%.2f', $cappedPricePerUnit),
                    'total_price' => (float) sprintf('%.2f', $cappedTotalPrice),
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
            }
        }
    }
}
