<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\Invoice;
use App\Models\PackingList;
use App\Models\Payment;
use App\Models\Client;
use App\Models\Transport;
use App\Models\Batch;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;

class OrderController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $model = 'Order';

        // Get paginated orders with all necessary relationships
        $query = Order::with([
            'buyer',
            'receiver',
            'transport',
            'payment',
            'invoice',
            'packingList',
            'batches' => function($query) {
                $query->with('product');
            }
        ]);

        // Search functionality
        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('id', 'like', "%{$search}%")
                    ->orWhere('status', 'like', "%{$search}%")
                    ->orWhereHas('buyer', function($q) use ($search) {
                        $q->where('name', 'like', "%{$search}%");
                    })
                    ->orWhereHas('receiver', function($q) use ($search) {
                        $q->where('name', 'like', "%{$search}%");
                    });
            });
        }

        // Sorting
        $sortBy = $request->get('sort_by', 'date');
        $sortDirection = $request->get('sort_direction', 'desc');
        $query->orderBy($sortBy, $sortDirection);

        // Pagination
        $perPage = $request->get('per_page', 10);
        $items = $query->paginate($perPage);

        // Foreign key map for the frontend
        $fkMap = [
            'buyer_id' => 'clients',
            'receiver_id' => 'clients',
            'transport_id' => 'transports',
            'payment_id' => 'payments',
        ];

        // Related options for form dropdowns
        $relatedOptions = [
            'buyer_id' => Client::select('id', 'name')->get()->toArray(),
            'receiver_id' => Client::select('id', 'name')->get()->toArray(),
            'transport_id' => Transport::select('id', 'name')->get()->toArray(),
        ];

        // If the request is for the OrderPage (custom view)
        if ($request->wantsJson() || $request->header('X-Inertia')) {
            return Inertia::render('OrderPage', [
                'model' => $model,
                'items' => $items,
                'filters' => [
                    'search' => $request->search,
                    'sort_by' => $sortBy,
                    'sort_direction' => $sortDirection,
                    'per_page' => $perPage,
                    'open_id' => $request->open_id,
                ],
                'fkMap' => $fkMap,
                'relatedOptions' => $relatedOptions,
            ]);
        }

        // Fallback to generic model page
        return Inertia::render('GenericModelPage', [
            'model' => $model,
            'items' => $items,
            'filters' => [
                'search' => $request->search,
                'sort_by' => $sortBy,
                'sort_direction' => $sortDirection,
                'per_page' => $perPage,
                'open_id' => $request->open_id,
            ],
            'fkMap' => $fkMap,
            'relatedOptions' => $relatedOptions,
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate(Order::getValidationRules());

        try {
            DB::beginTransaction();

            $order = Order::create($validated);

            // If batches are provided, attach them to the order
            if ($request->has('batches')) {
                foreach ($request->batches as $batchData) {
                    $order->batches()->attach($batchData['batch_id'], [
                        'quantity' => $batchData['quantity'],
                        'price_per_unit' => $batchData['price_per_unit'],
                        'total_price' => $batchData['quantity'] * $batchData['price_per_unit']
                    ]);
                }
            }

            DB::commit();

            return redirect()->route('orders.index')
                ->with('success', 'Order created successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Failed to create order: ' . $e->getMessage());
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Order $order)
    {
        $order->load([
            'buyer',
            'receiver',
            'transport',
            'payment',
            'invoice',
            'packingList',
            'batches' => function($query) {
                $query->with('product');
            }
        ]);

        return Inertia::render('OrderShow', [
            'order' => $order,
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Order $order)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Order $order)
    {
        $validated = $request->validate(Order::getValidationRules($order->id));

        try {
            DB::beginTransaction();

            $order->update($validated);

            // Update batches if provided
            if ($request->has('batches')) {
                $order->batches()->detach();
                foreach ($request->batches as $batchData) {
                    $order->batches()->attach($batchData['batch_id'], [
                        'quantity' => $batchData['quantity'],
                        'price_per_unit' => $batchData['price_per_unit'],
                        'total_price' => $batchData['quantity'] * $batchData['price_per_unit']
                    ]);
                }
            }

            DB::commit();

            return redirect()->route('orders.index')
                ->with('success', 'Order updated successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Failed to update order: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Order $order)
    {
        try {
            DB::beginTransaction();

            // Delete related records first
            $order->batches()->detach();

            // Delete the order
            $order->delete();

            DB::commit();

            return redirect()->route('orders.index')
                ->with('success', 'Order deleted successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Failed to delete order: ' . $e->getMessage());
        }
    }

    /**
     * Generate invoice for an order
     */
    public function generateInvoice(Order $order)
    {
        try {
            DB::beginTransaction();

            // Check if invoice already exists
            if ($order->invoice) {
                return back()->with('error', 'Invoice already exists for this order.');
            }

            // Calculate total amount from batches
            $totalAmount = $order->batches->sum(function($batch) {
                return $batch->pivot->quantity * $batch->pivot->price_per_unit;
            });

            // Create invoice
            $invoice = Invoice::create([
                'order_id' => $order->id,
                'invoice_date' => now(),
                'status' => 'draft',
                'total_amount' => $totalAmount,
            ]);

            // Create payment record if it doesn't exist
            if (!$order->payment) {
                $payment = Payment::create([
                    'order_id' => $order->id,
                    'amount' => $totalAmount,
                    'currency' => 'USD', // Default currency
                    'due_date' => now()->addDays(30),
                    'exchange_rate' => 1.0,
                    'payment_date' => null,
                    'payment_method' => 'bank_transfer',
                    'payment_status' => 'pending'
                ]);

                // Update order with payment_id
                $order->update(['payment_id' => $payment->id]);
            }

            DB::commit();

            return back()->with('success', 'Invoice generated successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Failed to generate invoice: ' . $e->getMessage());
        }
    }

    /**
     * Generate packing list for an order
     */
    public function generatePackingList(Order $order)
    {
        try {
            DB::beginTransaction();

            // Check if packing list already exists
            if ($order->packingList) {
                return back()->with('error', 'Packing list already exists for this order.');
            }

            // Create packing list
            $packingList = PackingList::create([
                'order_id' => $order->id,
                'packing_list_date' => now(),
                'status' => 'draft',
            ]);

            DB::commit();

            return back()->with('success', 'Packing list generated successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Failed to generate packing list: ' . $e->getMessage());
        }
    }

    /**
     * Add batch to order
     */
    public function addBatch(Request $request, Order $order)
    {
        $validated = $request->validate([
            'batch_id' => 'required|exists:batches,id',
            'quantity' => 'required|integer|min:1',
            'price_per_unit' => 'required|numeric|min:0',
        ]);
        try {
            $order->batches()->attach($validated['batch_id'], [
                'quantity' => $validated['quantity'],
                'price_per_unit' => $validated['price_per_unit'],
                'total_price' => $validated['quantity'] * $validated['price_per_unit']
            ]);

            return back()->with('success', 'Batch added to order successfully.');

        } catch (\Exception $e) {
            return back()->with('error', 'Failed to add batch to order: ' . $e->getMessage());
        }
    }

    /**
     * Remove batch from order
     */
    public function removeBatch(Order $order, Batch $batch)
    {
        try {
            $order->batches()->detach($batch->id);

            return back()->with('success', 'Batch removed from order successfully.');

        } catch (\Exception $e) {
            return back()->with('error', 'Failed to remove batch from order: ' . $e->getMessage());
        }
    }

    /**
     * Get available batches for order
     */
    public function getAvailableBatches(Order $order)
    {
        $availableBatches = Batch::whereDoesntHave('orders', function($query) use ($order) {
            $query->where('order_id', $order->id);
        })->with('product')->get();

        return response()->json($availableBatches);
    }

    /**
     * Update order status
     */
    public function updateStatus(Request $request, Order $order)
    {
        $validated = $request->validate([
            'status' => 'required|in:pending,confirmed,processing,shipped,delivered,cancelled'
        ]);

        try {
            $order->update(['status' => $validated['status']]);

            return back()->with('success', 'Order status updated successfully.');

        } catch (\Exception $e) {
            return back()->with('error', 'Failed to update order status: ' . $e->getMessage());
        }
    }
}
