from flask import Flask, jsonify, request
import sqlite3

app = Flask(__name__)

DB_PATH = "../wedding.db"


def get_db_connection():
    conn = sqlite3.connect(DB_PATH)
    conn.row_factory = sqlite3.Row
    return conn


@app.route("/")
def index():
    return jsonify(
        {
            "message": "Wedding Planner REST API Prototype",
            "endpoints": [
                "/weddings",
                "/weddings/<wedding_id>/events",
                "/weddings/<wedding_id>/guests",
                "/weddings/<wedding_id>/attendance",
                "/weddings/<wedding_id>/rsvp",
                "/availability/venue/<venue_id>?date=YYYY-MM-DD&start=HH:MM&end=HH:MM",
            ],
        }
    )


@app.route("/weddings", methods=["GET"])
def list_weddings():
    conn = get_db_connection()
    weddings = conn.execute("SELECT * FROM Wedding").fetchall()
    conn.close()
    return jsonify([dict(row) for row in weddings])


@app.route("/weddings/<int:wedding_id>/events", methods=["GET"])
def list_events(wedding_id):
    conn = get_db_connection()
    events = conn.execute(
        "SELECT * FROM Event WHERE wedding_id = ?", (wedding_id,)
    ).fetchall()
    conn.close()
    return jsonify([dict(row) for row in events])


@app.route("/weddings/<int:wedding_id>/guests", methods=["GET"])
def list_guests(wedding_id):
    conn = get_db_connection()
    guests = conn.execute(
        "SELECT * FROM Guest WHERE wedding_id = ?", (wedding_id,)
    ).fetchall()
    conn.close()
    return jsonify([dict(row) for row in guests])


@app.route("/weddings/<int:wedding_id>/attendance", methods=["GET"])
def attendance_overview(wedding_id):
    conn = get_db_connection()
    rows = conn.execute(
        """
        SELECT
            a.attendance_id,
            g.first_name || ' ' || g.last_name AS guest_name,
            a.status,
            a.role,
            a.table_number,
            e.event_type,
            e.date,
            e.start_time,
            e.end_time
        FROM Attendance a
        JOIN Guest g ON a.guest_id = g.guest_id
        JOIN Event e ON a.event_id = e.event_id
        WHERE g.wedding_id = ?
        ORDER BY e.date, e.start_time
        """,
        (wedding_id,),
    ).fetchall()
    conn.close()
    return jsonify([dict(row) for row in rows])


@app.route("/weddings/<int:wedding_id>/rsvp", methods=["GET"])
def rsvp_overview(wedding_id):
    conn = get_db_connection()
    rows = conn.execute(
        """
        SELECT
            r.rsvp_id,
            g.first_name || ' ' || g.last_name AS guest_name,
            e.event_type,
            r.status,
            r.response_date
        FROM Event_RSVP r
        JOIN Guest g ON r.guest_id = g.guest_id
        JOIN Event e ON r.event_id = e.event_id
        WHERE g.wedding_id = ?
        ORDER BY r.response_date
        """,
        (wedding_id,),
    ).fetchall()
    conn.close()
    return jsonify([dict(row) for row in rows])


@app.route("/availability/venue/<int:venue_id>", methods=["GET"])
def venue_availability(venue_id):
    date = request.args.get("date")
    start = request.args.get("start")
    end = request.args.get("end")

    if not date or not start or not end:
        return (
            jsonify(
                {"error": "Missing required parameters: date, start, end"}
            ),
            400,
        )

    conn = get_db_connection()
    conflicts = conn.execute(
        """
        SELECT * FROM Venue_Booking
        WHERE venue_id = ?
          AND date = ?
          AND status = 'CONFIRMED'
          AND NOT (end_time <= ? OR start_time >= ?)
        """,
        (venue_id, date, start, end),
    ).fetchall()
    conn.close()

    conflict_list = [dict(row) for row in conflicts]

    return jsonify(
        {
            "venue_id": venue_id,
            "date": date,
            "requested": {"start": start, "end": end},
            "available": len(conflict_list) == 0,
            "conflicts": conflict_list,
        }
    )


if __name__ == "__main__":
    app.run(debug=True)
