package it.finki.charitable.controller;

import it.finki.charitable.entities.*;
import it.finki.charitable.services.*;
import it.finki.charitable.util.FileUploadUtil;
import org.springframework.format.annotation.DateTimeFormat;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

@Controller
public class DonationPostController {

    private final DonationPostService donationPostService;
    private final UserService userService;
    private final FundsCollectedService fundsCollectedService;
    private final DonationInformationService donationInformationService;
    private final ReportPostService reportPostService;
    private final ReasonService reasonService;

    public DonationPostController(DonationPostService donationPostService, UserService userService, FundsCollectedService fundsCollectedService, DonationInformationService donationInformationService, ReportPostService reportPostService, ReasonService reasonService) {
        this.donationPostService = donationPostService;
        this.userService = userService;
        this.fundsCollectedService = fundsCollectedService;
        this.donationInformationService = donationInformationService;
        this.reportPostService = reportPostService;
        this.reasonService = reasonService;
    }

    @RequestMapping("/upload")
    public String upload() {
        return "upload";
    }

    @RequestMapping(value = "/newPost", method = RequestMethod.POST)
    public String newPost(Model model,
                          @RequestParam String title,
                          @RequestParam String fundsNeeded,
                          @RequestParam String currency,
                          @RequestParam String description,
                          @RequestParam String telekom,
                          @RequestParam String a1,
                          @RequestParam(defaultValue = "2021-01-01") @DateTimeFormat(iso = DateTimeFormat.ISO.DATE) LocalDate dateDue,
                          @RequestParam String bankAccount,
                          @RequestParam MultipartFile titleImage,
                          @RequestParam MultipartFile[] images,
                          @RequestParam MultipartFile[] moderatorImages) {

        DonationPost post = new DonationPost();
        post.setTitle(title);

        try {
            float funds = Float.parseFloat(fundsNeeded);
            post.setFundsNeeded(funds);
        } catch (NumberFormatException e) {
            e.printStackTrace();
        }

        post.setCurrency(currency);
        post.setDescription(description);
        post.setDateDue(dateDue);
        post.setBankAccount(bankAccount);
        post.setApproved(false);

        List<String> phoneNumbers = Arrays.asList(telekom, a1);

        List<String> photos = new ArrayList<>();
        photos.add(StringUtils.cleanPath(Objects.requireNonNull(titleImage.getOriginalFilename())));
        Arrays.stream(images).filter(i -> !i.isEmpty()).forEach(i -> photos.add(StringUtils.cleanPath(Objects.requireNonNull(i.getOriginalFilename()))));

        List<MultipartFile> files = new ArrayList<>();
        files.add(titleImage);
        files.addAll(Arrays.stream(images).filter(i -> !i.isEmpty()).collect(Collectors.toList()));

        List<String> moderatorPhotos = new ArrayList<>();
        Arrays.stream(moderatorImages).forEach(i -> moderatorPhotos.add(StringUtils.cleanPath(Objects.requireNonNull(i.getOriginalFilename()))));

        post.setPhoneNumbers(phoneNumbers);
        post.setImages(photos);
        post.setModeratorImages(moderatorPhotos);

        AppUser user = (AppUser) model.getAttribute("user");
        post.setUser(user);

        DonationPost savedPost = donationPostService.save(post);

        for (MultipartFile file : files) {
            String fileName = StringUtils.cleanPath(Objects.requireNonNull(file.getOriginalFilename()));
            String uploadDir = "post-photos/" + savedPost.getId();
            try {
                FileUploadUtil.saveFile(uploadDir, fileName, file);
            } catch (IOException e) {
                e.printStackTrace();
            }

        }

        for (MultipartFile file : moderatorImages) {

            String fileName = StringUtils.cleanPath(Objects.requireNonNull(file.getOriginalFilename()));
            String uploadDir = "moderator-photos/" + savedPost.getId();
            try {
                FileUploadUtil.saveFile(uploadDir, fileName, file);
            } catch (IOException e) {
                e.printStackTrace();
            }
        }

        return "upload";
    }

    @RequestMapping("/album")
    public String album(Model model) {
        List<DonationPost> postList = donationPostService.findAllByApproved(true);
        if (postList.size() == 0) {
            model.addAttribute("noPosts", true);
            return "album";
        }
        model.addAttribute("postList", postList);
        return "album";
    }

    @RequestMapping("/post")
    public String showPost(Model model, @RequestParam Long postid) {
        DonationPost post = donationPostService.getById(postid);
        if (post == null) {
            model.addAttribute("notFound", true);
            return "post";
        }

        if (post.getApproved() || (post.getUser().getUsername().equals(SecurityContextHolder.getContext().getAuthentication().getName()) && !post.getApproved())) {
            AppUser user = post.getUser();
            Moderator moderator = post.getModerator();
            model.addAttribute("post", post);
            model.addAttribute("createdByFirstName", user.getFirstName());
            model.addAttribute("createdByLastName", user.getLastName());
            if (moderator != null) {
                model.addAttribute("moderatorFirstName", moderator.getFirstName());
                model.addAttribute("moderatorLastName", moderator.getLastName());
            }
            double total = post.getFundsCollected().stream().mapToDouble(FundsCollected::getFunds).sum();
            model.addAttribute("total", total);
        } else {
            model.addAttribute("notFound", true);
        }
        return "post";
    }

    @RequestMapping("/donate")
    public String donate(Model model, @RequestParam Long postid,
                         @RequestParam String cardName,
                         @RequestParam String cardNumber,
                         @RequestParam String expiryDate,
                         @RequestParam String cvv,
                         @RequestParam float amount) {

        DonationPost post = donationPostService.getById(postid);
        FundsCollected funds = new FundsCollected("Online donation", amount);
        fundsCollectedService.save(funds);

        post.getFundsCollected().add(funds);
        donationPostService.save(post);

        DonationInformation donationInformation = new DonationInformation(amount, post.getId(), post.getTitle());
        donationInformationService.save(donationInformation);
        MainUser user = (MainUser) userService.loadUserByUsername(SecurityContextHolder.getContext().getAuthentication().getName());
        user.getDonationInformation().add(donationInformation);
        userService.saveUser(user);

        return String.format("redirect:/post?postid=%d", postid);
    }

    @RequestMapping("/report")
    public String report(@RequestParam Long postid,
                         @RequestParam String description) {

        DonationPost donationPost = donationPostService.getById(postid);
        ReportPost reportPost = reportPostService.findByDonationPost(donationPost);
        if(reportPost == null) {
            reportPost = new ReportPost();
            reportPost.setDonationPost(donationPost);
        }

        Reason reason = new Reason();
        AppUser user = (AppUser) SecurityContextHolder.getContext().getAuthentication().getPrincipal();
        reason.setUser(user);
        reason.setDescription(description);
        reasonService.save(reason);
        reportPost.getReasons().add(reason);
        reportPostService.save(reportPost);
        return String.format("redirect:/post?postid=%d", postid);
    }

    @ModelAttribute("user")
    public AppUser addAttributes() {
        if (SecurityContextHolder.getContext().getAuthentication().getPrincipal() != "anonymousUser") {
            return (AppUser) SecurityContextHolder.getContext().getAuthentication().getPrincipal();
        }
        return null;
    }
}
