using FinkiChattery.Api.Middleware;
using FinkiChattery.Api.Server;
using FinkiChattery.Api.Services;
using FinkiChattery.Commands.Questioning;
using FinkiChattery.Persistence.Context;
using FluentValidation.AspNetCore;
using Hangfire;
using Microsoft.AspNetCore.Builder;
using Microsoft.AspNetCore.Hosting;
using Microsoft.AspNetCore.Mvc;
using Microsoft.EntityFrameworkCore;
using Microsoft.Extensions.Configuration;
using Microsoft.Extensions.DependencyInjection;
using Microsoft.Extensions.Hosting;

namespace FinkiChattery.Api
{
    public class Startup
    {
        public Startup(IConfiguration configuration)
        {
            Configuration = configuration;
            AppSettings = new AppSettings(configuration);
        }

        public IConfiguration Configuration { get; }
        public AppSettings AppSettings { get; }

        public void ConfigureServices(IServiceCollection services)
        {
            services.AddSingletonServices();
            services.AddIdentityService(Configuration);
            services.AddMediator();
            services.AddEmailService();
            services.AddOriginUrlSettings();
            services.AddCurrentUser();
            services.AddRepos();
            services.AddAwsClient(Configuration);
            services.AddHangfireService(Configuration);

            services.AddDbContext<ApplicationDbContext>(options =>
                options.UseSqlServer(Configuration.GetConnectionString("DefaultConnection")));

            services.AddControllers()
                .ConfigureApiBehaviorOptions(options => options.InvalidModelStateResponseFactory = context => new BadRequestObjectResult(context.ModelState))
                .AddFluentValidation(fv => fv.RegisterValidatorsFromAssemblyContaining<AskQuestionValidator>());
            services.AddApiVersioning();

            services.AddCors(options =>
            {
                options.AddPolicy(
                    name: AppSettings.CorsSettings.CorsPolicyName,
                    builder => builder.WithOrigins(AppSettings.CorsSettings.AllowedCorsOrigins).AllowAnyHeader().AllowAnyMethod());
            });
        }

        public void Configure(IApplicationBuilder app, IWebHostEnvironment env)
        {
            if (env.IsDevelopment())
            {
                app.UseDeveloperExceptionPage();
            }

            app.UseHttpsRedirection();

            app.UseMiddleware<ExceptionHandler>();

            app.UseRouting();
            app.UseCors(AppSettings.CorsSettings.CorsPolicyName);

            app.UseAuthentication();

            app.UseAuthorization();

            app.UseEndpoints(endpoints =>
            {
                endpoints.MapControllerRoute("default", "{controller=Home}/{action=Index}/{id?}");
                endpoints.MapHangfireDashboard();
            });
        }
    }
}
