import { Injectable } from '@angular/core';
import { act, Actions, createEffect, ofType } from '@ngrx/effects';
import { catchError, filter, mergeMap, switchMap, withLatestFrom } from 'rxjs/operators';
import { PreviewQuestionsOrderEnum, SearchQuestionsQueryViewModel } from 'src/app/shared-app/models';
import { TranslateFromJsonService } from 'src/app/shared-app/services';

import { BaseApiService } from 'src/app/shared-app/services/base-api.service';
import { QuestionFacadeService } from '../question-facade.service';
import { VoteAnswerRequest } from './question-state-request.models';
import { PreviewQuestionResponse, QuestionStateResponse, VoteAnswerResponse } from './question-state-response.models';
import {
  EffectFinishedWorking,
  EffectFinishedWorkingError,
  GetPreviewQuestionsLatest,
  GetPreviewQuestionsLatestSuccess,
  GetPreviewQuestionsPopular,
  GetPreviewQuestionsPopularSuccess,
  GetQuestionState,
  GetQuestionStateSuccess,
  GetSearchQuestions,
  GetSearchQuestionsSuccess,
  QuestionActionTypes,
  VoteAnswer,
  VoteAnswerSuccess
} from './question.actions';
import { QuestionMapper } from './question.mapper';

@Injectable({
  providedIn: 'root'
})
export class QuestionEffects {
  constructor(
    private actions$: Actions,
    private api: BaseApiService,
    private translate: TranslateFromJsonService,
    private facade: QuestionFacadeService
  ) {}

  getQuestionState$ = createEffect(() => {
    return this.actions$.pipe(
      ofType<GetQuestionState>(QuestionActionTypes.GetQuestionState),
      switchMap((action) => {
        return this.api.get<QuestionStateResponse>(`v1/questions/${action.questionUid}`).pipe(
          switchMap((state) => [
            new GetQuestionStateSuccess(QuestionMapper.ToQuestionStateViewModel(state, this.translate)),
            new EffectFinishedWorking()
          ]),
          catchError((err) => [new EffectFinishedWorkingError(err)])
        );
      })
    );
  });

  getPreviewQuestionsLatest$ = createEffect(() => {
    return this.actions$.pipe(
      ofType<GetPreviewQuestionsLatest>(QuestionActionTypes.GetPreviewQuestionsLatest),
      withLatestFrom(this.facade.getPreviewQuestionsLatest()),
      filter(([action, questions]) => questions.length === 0),
      switchMap((action) => {
        return this.api.get<PreviewQuestionResponse[]>(`v1/questions/preview?order=${PreviewQuestionsOrderEnum.Latest}`).pipe(
          switchMap((state) => [
            new GetPreviewQuestionsLatestSuccess(QuestionMapper.ToPreviwQuestionsViewModel(state, this.translate)),
            new EffectFinishedWorking()
          ]),
          catchError((err) => [new EffectFinishedWorkingError(err)])
        );
      })
    );
  });

  getPreviewQuestionsPopular$ = createEffect(() => {
    return this.actions$.pipe(
      ofType<GetPreviewQuestionsPopular>(QuestionActionTypes.GetPreviewQuestionsPopular),
      withLatestFrom(this.facade.getPreviewQuestionsPopular()),
      filter(([action, questions]) => questions.length === 0),
      switchMap((action) => {
        return this.api.get<PreviewQuestionResponse[]>(`v1/questions/preview?order=${PreviewQuestionsOrderEnum.Popular}`).pipe(
          switchMap((state) => [
            new GetPreviewQuestionsPopularSuccess(QuestionMapper.ToPreviwQuestionsViewModel(state, this.translate)),
            new EffectFinishedWorking()
          ]),
          catchError((err) => [new EffectFinishedWorkingError(err)])
        );
      })
    );
  });

  getSearchQuestions$ = createEffect(() => {
    return this.actions$.pipe(
      ofType<GetSearchQuestions>(QuestionActionTypes.GetSearchQuestions),
      mergeMap((action) => {
        const categoriesAsString = action.categories !== null ? action.categories.join(',') : '';
        return this.api
          .get<PreviewQuestionResponse[]>(`v1/questions/search?searchText=${action.searchText}&categories=${categoriesAsString}`)
          .pipe(
            switchMap((state) => [
              new GetSearchQuestionsSuccess(
                QuestionMapper.ToPreviwQuestionsViewModel(state, this.translate),
                new SearchQuestionsQueryViewModel(action.searchText)
              ),
              new EffectFinishedWorking()
            ]),
            catchError((err) => [new EffectFinishedWorkingError(err)])
          );
      })
    );
  });

  voteAnswer$ = createEffect(() => {
    return this.actions$.pipe(
      ofType<VoteAnswer>(QuestionActionTypes.VoteAnswer),
      mergeMap((action) => {
        const body = new VoteAnswerRequest(action.voteType);
        return this.api.post<VoteAnswerResponse>(`v1/questions/${action.questionUid}/answers/${action.answerUid}/votes`, body).pipe(
          switchMap((state) => [new VoteAnswerSuccess(QuestionMapper.ToVoteAnswerViewModel(state)), new EffectFinishedWorking()]),
          catchError((err) => [new EffectFinishedWorkingError(err)])
        );
      })
    );
  });
}
