import { HttpErrorResponse } from '@angular/common/http';
import { Injectable } from '@angular/core';
import { Action, Store } from '@ngrx/store';
import { Observable, Subject } from 'rxjs';
import { filter } from 'rxjs/operators';

import {
  PreviewQuestionsOrderEnum,
  PreviewQuestionViewModel,
  QuestionStateViewModel,
  SearchQuestionsQueryViewModel,
  VoteType
} from 'src/app/shared-app/models';
import {
  EffectStartedWorking,
  GetPreviewQuestionsLatest,
  GetPreviewQuestionsPopular,
  GetQuestionState,
  GetSearchQuestions,
  VoteAnswer
} from './question-state/question.actions';
import { questionStateQuery } from './question-state/question.selectors';
import { QuestionState } from './question-state/question.state';

@Injectable({
  providedIn: 'root'
})
export class QuestionFacadeService {
  private questionSearchedSubject = new Subject<any>();

  effectWorking$: Observable<boolean | HttpErrorResponse>;

  constructor(private store: Store<QuestionState>) {
    this.effectWorking$ = this.store.select(questionStateQuery.effectWorking).pipe(filter((effect) => effect !== null));
  }

  public questionSearched(): Observable<void> {
    return this.questionSearchedSubject.asObservable();
  }

  public getQuestion(): Observable<QuestionStateViewModel> {
    return this.store
      .select(questionStateQuery.getQuestion)
      .pipe(filter((x: QuestionStateViewModel | null): x is QuestionStateViewModel => x !== null));
  }

  public getSearchQuestions(): Observable<PreviewQuestionViewModel[]> {
    return this.store.select(questionStateQuery.getSearchQuestions);
  }

  public getSearchQuestionsQuery(): Observable<SearchQuestionsQueryViewModel> {
    return this.store
      .select(questionStateQuery.getSearchQuestionsQuery)
      .pipe(filter((x: SearchQuestionsQueryViewModel | null): x is SearchQuestionsQueryViewModel => x !== null));
  }

  public getPreviewQuestionsLatest(): Observable<PreviewQuestionViewModel[]> {
    return this.store.select(questionStateQuery.getPreviewQuestionsLatest);
  }

  public getPreviewQuestionsPopular(): Observable<PreviewQuestionViewModel[]> {
    return this.store.select(questionStateQuery.getPreviewQuestionsPopular);
  }

  public fetchPreviewQuestions(orderBy: PreviewQuestionsOrderEnum): void {
    if (orderBy === PreviewQuestionsOrderEnum.Latest) {
      this.fetchPreviewQuestionsLatest();
    } else if (orderBy === PreviewQuestionsOrderEnum.Popular) {
      this.fetchPreviewQuestionsPopular();
    }
  }

  public voteAnswer(answerUid: string, questionUid: string, voteType: VoteType): void {
    this.dispatchEffect(new VoteAnswer(questionUid, answerUid, voteType));
  }

  public fetchQuestion(questionUid: string): void {
    this.dispatchEffect(new GetQuestionState(questionUid));
  }

  public searchQuestions(searchText: string, categories: string[]): void {
    this.dispatchEffect(new GetSearchQuestions(searchText, categories));
    this.questionSearchedSubject.next(Math.random());
  }

  private fetchPreviewQuestionsLatest(): void {
    this.dispatchEffect(new GetPreviewQuestionsLatest());
  }

  private fetchPreviewQuestionsPopular(): void {
    this.dispatchEffect(new GetPreviewQuestionsPopular());
  }

  private dispatch(action: Action): void {
    this.store.dispatch(action);
  }

  private dispatchEffect(action: Action): void {
    this.dispatch(new EffectStartedWorking());
    this.dispatch(action);
  }
}
