import { Injectable } from '@angular/core';
import { act, Actions, createEffect, ofType } from '@ngrx/effects';
import { catchError, filter, mergeMap, switchMap, withLatestFrom } from 'rxjs/operators';
import { PreviewQuestionsOrderEnum, SearchQuestionsQueryViewModel } from 'src/app/shared-app/models';
import { TranslateFromJsonService } from 'src/app/shared-app/services';

import { BaseApiService } from 'src/app/shared-app/services/base-api.service';
import { QuestionFacadeService } from '../question-facade.service';
import { PreviewQuestionResponse, QuestionStateResponse } from './question-state.models';
import {
  EffectFinishedWorking,
  EffectFinishedWorkingError,
  GetPreviewQuestionsLatest,
  GetPreviewQuestionsLatestSuccess,
  GetPreviewQuestionsPopular,
  GetPreviewQuestionsPopularSuccess,
  GetQuestionState,
  GetQuestionStateSuccess,
  GetSearchQuestions,
  GetSearchQuestionsSuccess,
  QuestionActionTypes
} from './question.actions';
import { QuestionMapper } from './question.mapper';

@Injectable({
  providedIn: 'root'
})
export class QuestionEffects {
  constructor(
    private actions$: Actions,
    private api: BaseApiService,
    private translate: TranslateFromJsonService,
    private facade: QuestionFacadeService
  ) {}

  getQuestionState$ = createEffect(() => {
    return this.actions$.pipe(
      ofType<GetQuestionState>(QuestionActionTypes.GetQuestionState),
      switchMap((action) => {
        return this.api.get<QuestionStateResponse>(`v1/questions/${action.questionUid}`).pipe(
          switchMap((state) => [new GetQuestionStateSuccess(QuestionMapper.ToQuestionStateViewModel(state)), new EffectFinishedWorking()]),
          catchError((err) => [new EffectFinishedWorkingError(err)])
        );
      })
    );
  });

  getPreviewQuestionsLatest$ = createEffect(() => {
    return this.actions$.pipe(
      ofType<GetPreviewQuestionsLatest>(QuestionActionTypes.GetPreviewQuestionsLatest),
      withLatestFrom(this.facade.getPreviewQuestionsLatest()),
      filter(([action, questions]) => questions.length === 0),
      switchMap((action) => {
        return this.api.get<PreviewQuestionResponse[]>(`v1/questions/preview?order=${PreviewQuestionsOrderEnum.Latest}`).pipe(
          switchMap((state) => [
            new GetPreviewQuestionsLatestSuccess(QuestionMapper.ToPreviwQuestionsViewModel(state, this.translate)),
            new EffectFinishedWorking()
          ]),
          catchError((err) => [new EffectFinishedWorkingError(err)])
        );
      })
    );
  });

  getPreviewQuestionsPopular$ = createEffect(() => {
    return this.actions$.pipe(
      ofType<GetPreviewQuestionsPopular>(QuestionActionTypes.GetPreviewQuestionsPopular),
      withLatestFrom(this.facade.getPreviewQuestionsPopular()),
      filter(([action, questions]) => questions.length === 0),
      switchMap((action) => {
        return this.api.get<PreviewQuestionResponse[]>(`v1/questions/preview?order=${PreviewQuestionsOrderEnum.Popular}`).pipe(
          switchMap((state) => [
            new GetPreviewQuestionsPopularSuccess(QuestionMapper.ToPreviwQuestionsViewModel(state, this.translate)),
            new EffectFinishedWorking()
          ]),
          catchError((err) => [new EffectFinishedWorkingError(err)])
        );
      })
    );
  });

  getSearchQuestions$ = createEffect(() => {
    return this.actions$.pipe(
      ofType<GetSearchQuestions>(QuestionActionTypes.GetSearchQuestions),
      mergeMap((action) => {
        const categoriesAsString = action.categories !== null ? action.categories.join(',') : '';
        return this.api
          .get<PreviewQuestionResponse[]>(`v1/questions/search?searchText=${action.searchText}&categories=${categoriesAsString}`)
          .pipe(
            switchMap((state) => [
              new GetSearchQuestionsSuccess(
                QuestionMapper.ToPreviwQuestionsViewModel(state, this.translate),
                new SearchQuestionsQueryViewModel(action.searchText)
              ),
              new EffectFinishedWorking()
            ]),
            catchError((err) => [new EffectFinishedWorkingError(err)])
          );
      })
    );
  });
}
