﻿using FinkiChattery.Common.Mediator.Contracs;
using FinkiChattery.Common.Mediator.Interfaces;
using FinkiChattery.Common.User;
using FinkiChattery.Persistence.Models;
using FinkiChattery.Persistence.Repositories.Contracts;
using FinkiChattery.Persistence.UnitOfWork;
using System;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;

namespace FinkiChattery.Commands.Questioning
{
    public class AnswerQuestionCommand : ICommand<AnswerQuestionStateDto>
    {
        public AnswerQuestionCommand(Guid questionUid, string text)
        {
            QuestionUid = questionUid;
            Text = text;
        }

        public Guid QuestionUid { get; }
        public string Text { get; }
    }

    public class AnswerQuestionHandler : ICommandHandler<AnswerQuestionCommand, AnswerQuestionStateDto>
    {
        public AnswerQuestionHandler(IUnitOfWork unitOfWork, ICurrentUser currentUser, IEventService eventService)
        {
            UnitOfWork = unitOfWork;
            CurrentUser = currentUser;
            EventService = eventService;
        }

        public IUnitOfWork UnitOfWork { get; }
        public ICurrentUser CurrentUser { get; }
        public IEventService EventService { get; }

        public async Task<AnswerQuestionStateDto> Handle(AnswerQuestionCommand request, CancellationToken cancellationToken)
        {
            var question = await UnitOfWork.Questions.GetByUidAsync(request.QuestionUid);
            var student = await UnitOfWork.Students.GetStudent(CurrentUser.Id);

            var answer = new Answer()
            {
                Text = request.Text,
                QuestionFk = question.Id,
                StudentFk = student.Id,
            };

            UnitOfWork.Answers.Add(answer);
            await UnitOfWork.SaveAsync();

            EventService.Enqueue(new QuestionAnsweredEvent(question.Uid, answer.Uid, student.Uid));

            return new AnswerQuestionStateDto(answer.Id,
                                              answer.Uid,
                                              answer.Text,
                                              answer.CorrectAnswer,
                                              answer.CreatedOn,
                                              answer.VotesCount,
                                              new AnswerStudentQuestionStateDto(student.Id,
                                                                                student.Uid,
                                                                                student.IndexNumber,
                                                                                student.ImageUrl,
                                                                                student.Reputation),
                                              Enumerable.Empty<AnswerResponseQuestionStateDto>());
        }
    }
}
