import { Injectable } from '@angular/core';
import { Actions, createEffect, ofType } from '@ngrx/effects';
import { catchError, mergeMap, switchMap } from 'rxjs/operators';
import { TranslateFromJsonService } from 'src/app/shared-app/services';

import { BaseApiService } from 'src/app/shared-app/services/base-api.service';
import { CreateCategoryRequest } from './category-state-request.models';
import { CategoryStateResponse } from './category-state.models';
import {
  AddCategory,
  AddCategorySuccess,
  CategoryActionTypes,
  EffectFinishedWorking,
  EffectFinishedWorkingError,
  GetCategoriesState,
  GetCategoriesStateSuccess
} from './category.actions';
import { CategoriesMapper } from './category.mapper';

@Injectable({
  providedIn: 'root'
})
export class CategoriesEffects {
  constructor(private actions$: Actions, private api: BaseApiService, private translate: TranslateFromJsonService) {}

  getCategoriesState$ = createEffect(() => {
    return this.actions$.pipe(
      ofType<GetCategoriesState>(CategoryActionTypes.GetCategoriesState),
      switchMap((action) => {
        return this.api.get<CategoryStateResponse[]>(`v1/categories`).pipe(
          switchMap((state) => [
            new GetCategoriesStateSuccess(CategoriesMapper.ToCategoriesStateViewModel(state, this.translate)),
            new EffectFinishedWorking()
          ]),
          catchError((err) => [new EffectFinishedWorkingError(err)])
        );
      })
    );
  });

  addCategory$ = createEffect(() => {
    return this.actions$.pipe(
      ofType<AddCategory>(CategoryActionTypes.AddCategory),
      mergeMap((action) => {
        return this.api.post<string>(`v1/categories`, new CreateCategoryRequest(action.name)).pipe(
          switchMap((state) => [
            new AddCategorySuccess(CategoriesMapper.ToCategoryStateViewModel(state, action.name, this.translate)),
            new EffectFinishedWorking()
          ]),
          catchError((err) => [new EffectFinishedWorkingError(err)])
        );
      })
    );
  });
}
