﻿using FinkiChattery.Common.Mediator.Contracs;
using FinkiChattery.Common.Mediator.Interfaces;
using FinkiChattery.Common.User;
using FinkiChattery.Persistence.Helpers;
using FinkiChattery.Persistence.Models;
using FinkiChattery.Persistence.UnitOfWork;
using System;
using System.Threading;
using System.Threading.Tasks;

namespace FinkiChattery.Commands.Questioning
{
    public class VoteAnswerCommand : ICommand<VoteAnswerDto>
    {
        public VoteAnswerCommand(VoteType voteType, Guid answerUid, Guid questionUid)
        {
            VoteType = voteType;
            AnswerUid = answerUid;
            QuestionUid = questionUid;
        }

        public VoteType VoteType { get; }
        public Guid AnswerUid { get; }
        public Guid QuestionUid { get; }
    }

    public class VoteAnswerHandler : ICommandHandler<VoteAnswerCommand, VoteAnswerDto>
    {
        public VoteAnswerHandler(IUnitOfWork unitOfWork, ICurrentUser currentUser, IEventService eventService)
        {
            UnitOfWork = unitOfWork;
            CurrentUser = currentUser;
            EventService = eventService;
        }

        public IUnitOfWork UnitOfWork { get; }
        public ICurrentUser CurrentUser { get; }
        public IEventService EventService { get; }

        public async Task<VoteAnswerDto> Handle(VoteAnswerCommand request, CancellationToken cancellationToken)
        {
            var student = await UnitOfWork.Students.GetStudent(CurrentUser.Id);
            var answer = await UnitOfWork.Answers.GetByUidAsync(request.AnswerUid);
            var vote = await UnitOfWork.Votes.GetVoteForAnswerByStudent(request.AnswerUid, CurrentUser.Id);
            var voteAlreadyExists = false;

            if (vote == null)
            {
                vote = new Vote()
                {
                    AnswerFk = answer.Id,
                    StudentFk = student.Id,
                    VoteType = request.VoteType
                };

                UnitOfWork.Votes.Add(vote);
            } 
            else
            {
                vote.VoteType = request.VoteType;
                voteAlreadyExists = true;
            }

            await UnitOfWork.SaveAsync();

            EventService.Enqueue(new AnswerVotedEvent(request.QuestionUid, request.AnswerUid, student.Uid, request.VoteType, voteAlreadyExists));

            return new VoteAnswerDto(vote.Uid, voteAlreadyExists);
        }
    }
}
