# MySQL to PostgreSQL Migration Guide

This guide will help you migrate the TruckSimulator application from MySQL to PostgreSQL.

## Prerequisites

1. **PostgreSQL Installation**
   - Install PostgreSQL 12+ on your system
   - Ensure PostgreSQL service is running
   - Default port: 5432

2. **Database Setup**
   - Create a database named `trucksimulator`
   - Create a user `postgres` with password `2405` (or update connection string)
   - Grant necessary permissions

## Migration Steps

### 1. Install Dependencies

```bash
cd ds-db-ws
npm install
```

This will install the PostgreSQL driver (`pg`) and remove the MySQL dependency.

### 2. Database Migration

Run the migration script to create PostgreSQL tables:

```bash
node migrate-to-postgres.js
```

This script will:
- Convert MySQL syntax to PostgreSQL
- Create all necessary tables
- Handle data type conversions
- Set up proper constraints

### 3. Manual Database Setup (Alternative)

If the migration script doesn't work, you can manually:

1. Create the database:
```sql
CREATE DATABASE trucksimulator;
```

2. Import the converted SQL file (after manual conversion):
```bash
psql -U postgres -d trucksimulator -f converted_schema.sql
```

### 4. Update Connection String (if needed)

If your PostgreSQL setup differs, update `models/db.js`:

```javascript
const pool = new Pool({
  host: 'localhost',        // Your PostgreSQL host
  user: 'postgres',         // Your PostgreSQL user
  password: '2405',         // Your PostgreSQL password
  database: 'trucksimulator', // Your database name
  port: 5432,              // Your PostgreSQL port
  max: 10,
  idleTimeoutMillis: 30000,
  connectionTimeoutMillis: 2000,
});
```

### 5. Test the Application

```bash
npm start
```

Visit `http://localhost:3000` and test:
- User registration/login
- Product browsing
- Procurement functionality
- Admin dashboard

## Key Changes Made

### 1. Database Driver
- **Before**: `mysql2` package
- **After**: `pg` package

### 2. Connection Configuration
- **Before**: MySQL connection pool
- **After**: PostgreSQL connection pool

### 3. Query Syntax
- **Before**: `?` placeholders (`SELECT * FROM users WHERE id = ?`)
- **After**: `$1, $2, $3...` placeholders (`SELECT * FROM users WHERE id = $1`)

### 4. Result Handling
- **Before**: `const [rows] = await pool.query(...)`
- **After**: `const result = await pool.query(...); const rows = result.rows`

### 5. SQL Functions
- **Before**: `TIMESTAMPDIFF(MINUTE, start, end)`
- **After**: `EXTRACT(EPOCH FROM (end - start))/60`

- **Before**: `JSON_OBJECT('key', value)`
- **After**: `JSON_BUILD_OBJECT('key', value)`

### 6. Data Types
- **Before**: `AUTO_INCREMENT`
- **After**: `SERIAL`

- **Before**: `DATETIME`
- **After**: `TIMESTAMP`

## Troubleshooting

### Common Issues

1. **Connection Refused**
   - Ensure PostgreSQL is running
   - Check port 5432 is open
   - Verify user credentials

2. **Permission Denied**
   - Grant proper permissions to the database user
   - Check database ownership

3. **Syntax Errors**
   - Some MySQL-specific functions may need manual conversion
   - Check PostgreSQL documentation for equivalent functions

4. **Data Type Issues**
   - Review column types after migration
   - Some MySQL types may need adjustment

### Verification Commands

Test database connection:
```bash
psql -U postgres -d trucksimulator -c "SELECT NOW();"
```

Check tables:
```bash
psql -U postgres -d trucksimulator -c "\dt"
```

## Rollback Plan

If you need to rollback to MySQL:

1. Revert `package.json`:
```json
{
  "dependencies": {
    "mysql2": "^3.14.1"
  }
}
```

2. Revert `models/db.js` to MySQL configuration

3. Revert all query syntax changes

4. Run `npm install` to restore MySQL driver

## Support

If you encounter issues during migration:

1. Check PostgreSQL logs
2. Verify all dependencies are installed
3. Test database connection manually
4. Review error messages for specific syntax issues

The migration script includes error handling and will show warnings for problematic statements that may need manual adjustment.
