#!/usr/bin/env node

/**
 * Database Migration Script: MySQL to PostgreSQL
 * 
 * This script helps migrate the TruckSimulator database from MySQL to PostgreSQL.
 * 
 * Prerequisites:
 * 1. PostgreSQL server running on localhost:5432
 * 2. PostgreSQL user 'postgres' with password '2405'
 * 3. Database 'TruckSimulator' created in PostgreSQL
 * 
 * Usage:
 * node migrate-to-postgres.js
 */

const { Pool } = require('pg');
const fs = require('fs');
const path = require('path');

// PostgreSQL connection
const pool = new Pool({
  host: 'localhost',
  user: 'postgres',
  password: 'Marconioni123',
  database: 'TruckSimulator',
  port: 5432,
});

async function createTables() {
  try {
    console.log('Creating PostgreSQL tables...');
    
    // Read the SQL file
    const sqlFile = path.join(__dirname, '..', 'sql data base', 'TruckSimulator.sql');
    const sqlContent = fs.readFileSync(sqlFile, 'utf8');
    
    // Convert MySQL syntax to PostgreSQL
    let postgresSQL = sqlContent
      // Replace AUTO_INCREMENT with SERIAL
      .replace(/AUTO_INCREMENT/g, 'SERIAL')
      // Replace MySQL data types
      .replace(/INT\(\d+\)/g, 'INTEGER')
      .replace(/VARCHAR\(\d+\)/g, 'VARCHAR')
      .replace(/DATETIME/g, 'TIMESTAMP')
      // Replace MySQL specific syntax
      .replace(/ENGINE=InnoDB/g, '')
      .replace(/DEFAULT CHARSET=utf8mb4/g, '')
      .replace(/COLLATE utf8mb4_0900_ai_ci/g, '')
      // Remove MySQL specific comments and syntax
      .replace(/--.*$/gm, '')
      .replace(/\/\*.*?\*\//gs, '');
    
    // Split into individual statements
    const statements = postgresSQL
      .split(';')
      .map(stmt => stmt.trim())
      .filter(stmt => stmt.length > 0 && !stmt.startsWith('--'));
    
    // Execute each statement
    for (const statement of statements) {
      if (statement.trim()) {
        try {
          await pool.query(statement);
          console.log('✓ Executed:', statement.substring(0, 50) + '...');
        } catch (err) {
          console.warn('⚠ Warning executing statement:', err.message);
          console.log('Statement:', statement.substring(0, 100) + '...');
        }
      }
    }
    
    console.log('✓ Database tables created successfully!');
    
  } catch (err) {
    console.error('❌ Error creating tables:', err);
    throw err;
  }
}

async function testConnection() {
  try {
    console.log('Testing PostgreSQL connection...');
    const result = await pool.query('SELECT NOW() as current_time');
    console.log('✓ Connection successful! Current time:', result.rows[0].current_time);
    return true;
  } catch (err) {
    console.error('❌ Connection failed:', err.message);
    return false;
  }
}

async function main() {
  console.log('🚀 Starting MySQL to PostgreSQL migration...\n');
  
  // Test connection
  const connected = await testConnection();
  if (!connected) {
    console.log('\n❌ Cannot proceed without database connection.');
    console.log('Please ensure PostgreSQL is running and the database exists.');
    process.exit(1);
  }
  
  try {
    // Create tables
    await createTables();
    
    console.log('\n✅ Migration completed successfully!');
    console.log('\nNext steps:');
    console.log('1. Install dependencies: npm install');
    console.log('2. Start the application: npm start');
    console.log('3. Test all functionality to ensure everything works');
    
  } catch (err) {
    console.error('\n❌ Migration failed:', err);
    process.exit(1);
  } finally {
    await pool.end();
  }
}

// Run the migration
if (require.main === module) {
  main().catch(console.error);
}

module.exports = { createTables, testConnection };
