const { v4: uuidv4 } = require('uuid');
const pool = require('../models/db');

exports.createRequest = async (req, res) => {
  const { items, cardId, cvv } = req.body;
  const customerId = req.session?.user?.id;

  if (!customerId || req.session.user.type !== 'customer') {
    return res.status(403).json({ error: 'Unauthorized' });
  }

  if (!Array.isArray(items) || items.length === 0) {
    return res.status(400).json({ error: 'Cart is empty' });
  }

  try {
    
    const cardResult = await pool.query("SELECT Balance, CVV FROM wallet WHERE WalletID = $1", [cardId]);
    if (cardResult.rows.length === 0) return res.status(400).json({ error: "Invalid card." });
    if (cardResult.rows[0].cvv !== cvv) return res.status(400).json({ error: "Invalid CVV." });

  
    let totalCost = 0;
    const preparedItems = [];

    for (const item of items) {
      const productResult = await pool.query("SELECT Price FROM product WHERE ProductID = $1", [item.productId]);
      if (productResult.rows.length === 0) return res.status(400).json({ error: "Invalid product." });

      const price = parseFloat(productResult.rows[0].price);
      let monthlyPay = null;
      let totalPrice = price;
      let duration = null;

      if (item.type === 'Rent') {
        duration = parseInt(item.duration || 24);
        monthlyPay = parseFloat((price / 36).toFixed(2));
        totalPrice = parseFloat((monthlyPay * duration).toFixed(2));
      }

      totalCost += totalPrice;
      preparedItems.push({ ...item, monthlyPay, totalPrice, duration });
    }

   
    if (cardResult.rows[0].balance < totalCost) {
      return res.status(400).json({ error: "Insufficient balance." });
    }

    
    await pool.query("UPDATE wallet SET Balance = Balance - $1 WHERE WalletID = $2", [totalCost, cardId]);

    
    const groupId = uuidv4();

    for (const item of preparedItems) {
      await pool.query(`
        INSERT INTO procurement_request 
        (CustomerID, ProductID, Quantity, RequestedAt, Status, PaymentStatus, MonthlyPay, TotalPrice, Duration, CardID, TransactionType, PaymentMethod, GroupID)
        VALUES ($1, $2, $3, NOW(), 'Pending', 'Reserved', $4, $5, $6, $7, $8, $9, $10)`,
        [
          customerId,
          item.productId,
          item.quantity || 1,
          item.monthlyPay,
          item.totalPrice,
          item.duration,
          cardId,
          item.type || 'Buy',
          'Card',
          groupId
        ]
      );
    }

    res.json({ success: true });
  } catch (err) {
    console.error("Create request error:", err);
    res.status(500).json({ error: 'Failed to create request' });
  }
};




exports.getPendingRequests = async (req, res) => {
  try {
    const result = await pool.query(`
      SELECT pr.GroupID, pr.RequestID, c.CustomerName, c.CustomerSurName,
             p.Model, pr.Quantity, pr.TotalPrice, pr.Duration
      FROM procurement_request pr
      JOIN customer c ON c.CustomerID = pr.CustomerID
      JOIN product p ON p.ProductID = pr.ProductID
      WHERE pr.Status = 'Pending'
      ORDER BY pr.GroupID, pr.RequestID
    `);

    
    const grouped = {};
    for (const row of result.rows) {
      if (!grouped[row.groupid]) grouped[row.groupid] = [];
      grouped[row.groupid].push(row);
    }

    res.json(grouped);
  } catch (err) {
    console.error("Error fetching pending grouped requests:", err);
    res.status(500).json({ error: "Server error" });
  }
};
