const productModel = require('../models/productModel');
const pool = require('../models/db');

exports.getAll = async (req, res) => {
  try {
    const products = await productModel.getAll();
    res.json(products);
  } catch (err) {
    console.error(err);
    res.status(500).send('Server error');
  }
};
exports.getAllProducts = async (req, res) => {
  try {
    console.log("Session user:", req.session.user);

    let query = `
      SELECT 
        p.productid, p.model, p.price, p.licenseplate, p.status,
        CASE 
          WHEN t.productid IS NOT NULL THEN 'truck'
          WHEN tr.productid IS NOT NULL THEN 'trailer'
          ELSE 'unknown'
        END AS type,
        JSON_BUILD_OBJECT('HP', t.hp, 'Capacity', tr.capacity) AS specs
      FROM product p
      LEFT JOIN truck t ON p.productid = t.productid
      LEFT JOIN trailer tr ON p.productid = tr.productid
    `;

    if (!req.session.user || req.session.user.type !== "employee") {
      query += " WHERE p.status = 'available'";
    }

    const result = await pool.query(query);
    const products = result.rows;

    const parsed = products.map(p => ({
      ProductID: p.productid,
      Model: p.model,
      Price: p.price,
      LicensePlate: p.licenseplate,
      Status: p.status,
      type: p.type,
      specs: typeof p.specs === 'string' ? JSON.parse(p.specs) : p.specs
    }));

    res.json(parsed);
  } catch (err) {
    console.error("Error loading products:", err);
    res.status(500).json({ error: "Failed to fetch products" });
  }
};



exports.getAvailable = (req, res) => {
  Product.getAvailable((err, results) => {
    if (err) return res.status(500).send('Error fetching available products');
    res.json(results);
  });
};

exports.getById = (req, res) => {
  Product.getById(req.params.id, (err, result) => {
    if (err) return res.status(500).send('Error fetching product');
    res.json(result);
  });
};

exports.addProduct = async (req, res) => {
  const { Type, Model, Price, LicensePlate, Status, HP, Capacity } = req.body;

  try {
    const idResult = await pool.query("SELECT MAX(productid) + 1 AS newid FROM product");
    const newID = idResult.rows[0].newid || 1;

    await pool.query(
      `INSERT INTO product (productid, model, price, licenseplate, status)
       VALUES ($1, $2, $3, $4, $5)`,
      [newID, Model, Price, LicensePlate, Status]
    );

    if (Type === "truck") {
      await pool.query("INSERT INTO truck (productid, hp) VALUES ($1, $2)", [newID, HP]);
    } else if (Type === "trailer") {
      await pool.query("INSERT INTO trailer (productid, capacity) VALUES ($1, $2)", [newID, Capacity]);
    }

    res.json({ message: "Product added", id: newID });
  } catch (err) {
    console.error("Add product error:", err);
    res.status(500).send("Error adding product");
  }
};
exports.updateProduct = async (req, res) => {
  const { Model, Price, LicensePlate, Status, HP, Capacity } = req.body;
  const id = req.params.id;

  if (!id) {
    return res.status(400).json({ error: "Product ID is required" });
  }

  try {
    // Convert empty string to null for numeric fields
    const price = Price === '' ? null : parseFloat(Price);

    const client = await pool.connect();
    try {
      await client.query('BEGIN');

      await client.query(
        `UPDATE product SET Model=$1, Price=$2, LicensePlate=$3, Status=$4 WHERE ProductID=$5`,
        [Model, price, LicensePlate, Status, id]
      );

      // Optionally update truck/trailer specs if provided
      if (HP !== undefined && HP !== null && HP !== '') {
        await client.query(
          `INSERT INTO truck (ProductID, HP) VALUES ($1, $2)
           ON CONFLICT (ProductID) DO UPDATE SET HP = EXCLUDED.HP`,
          [id, parseInt(HP)]
        );
      }

      if (Capacity !== undefined && Capacity !== null && Capacity !== '') {
        await client.query(
          `INSERT INTO trailer (ProductID, Capacity) VALUES ($1, $2)
           ON CONFLICT (ProductID) DO UPDATE SET Capacity = EXCLUDED.Capacity`,
          [id, parseFloat(Capacity)]
        );
      }

      await client.query('COMMIT');
      res.json({ message: "Product updated" });
    } catch (txErr) {
      await client.query('ROLLBACK');
      throw txErr;
    } finally {
      client.release();
    }
  } catch (err) {
    console.error("Error updating product:", err);
    res.status(500).send("Error updating product");
  }
};

exports.deleteProduct = async (req, res) => {
  const id = req.params.id;

  if (!id) {
    return res.status(400).json({ error: "Product ID is required" });
  }

  try {
    const client = await pool.connect();
    try {
      await client.query('BEGIN');

      // Remove dependent rows first to satisfy foreign key constraints
      await client.query("DELETE FROM truck WHERE ProductID = $1", [id]);
      await client.query("DELETE FROM trailer WHERE ProductID = $1", [id]);
      await client.query("DELETE FROM views WHERE ProductID = $1", [id]);
      await client.query("DELETE FROM maintenance WHERE ProductID = $1", [id]);
      await client.query("DELETE FROM customerfeedback WHERE ProductID = $1", [id]);
      await client.query(
        "DELETE FROM t_type WHERE TransactionID IN (SELECT TransactionID FROM procurement WHERE ProductID = $1)",
        [id]
      );
      await client.query("DELETE FROM procurement_request WHERE ProductID = $1", [id]);
      await client.query("DELETE FROM procurement WHERE ProductID = $1", [id]);

      // Finally delete the product
      await client.query("DELETE FROM product WHERE ProductID = $1", [id]);

      await client.query('COMMIT');
      res.json({ message: "Product deleted" });
    } catch (txErr) {
      await client.query('ROLLBACK');
      throw txErr;
    } finally {
      client.release();
    }
  } catch (err) {
    console.error("Error deleting product:", err);
    res.status(500).send("Error deleting product");
  }
};
