package com.example.rezevirajmasa.demo.service.impl;

import com.example.rezevirajmasa.demo.model.Restaurant;
import com.example.rezevirajmasa.demo.model.TableEntity;
import com.example.rezevirajmasa.demo.model.exceptions.InvalidRestaurantIdException;
import com.example.rezevirajmasa.demo.repository.RestaurantRepository;
import com.example.rezevirajmasa.demo.repository.TableRepository;
import com.example.rezevirajmasa.demo.service.RestaurantService;
import com.example.rezevirajmasa.demo.service.TableService;
import com.sun.tools.jconsole.JConsoleContext;
import jakarta.transaction.Transactional;
import org.springframework.stereotype.Service;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.math.BigDecimal;
import java.time.format.DateTimeParseException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;


@Service
@Transactional
public class RestaurantServiceImpl implements RestaurantService {
    private final RestaurantRepository restaurantRepository;
    private final TableRepository tableRepository;
    private final TableService tableService;

    public RestaurantServiceImpl(RestaurantRepository restaurantRepository, TableRepository tableRepository, TableService tableService) {
        this.restaurantRepository = restaurantRepository;
        this.tableRepository = tableRepository;
        this.tableService = tableService;
    }

    @Override
    public List<Restaurant> listall() {
        return restaurantRepository.findAll();
    }

//    @Override
//    public void save(String name, String cuisineType, String address, String phone, String operatingHours, String website, String socialMediaLinks, BigDecimal rating, List<Long> tablesList) {
//        List<TableEntity> tableEntities = tableRepository.findAllById(tablesList);
//        restaurantRepository.save(new Restaurant(name, cuisineType, address, phone, operatingHours, website, socialMediaLinks, rating, tableEntities));
//    }

    @Override
    public void save(Restaurant restaurant, int numberOfTables, List<Integer> tableCapacities, List<String> tableLocations, List<String> tableSmokingAreas, List<String> tableDescriptions) {
        restaurantRepository.save(restaurant);
        String[] hours = restaurant.getOperatingHours().split("-");
        try {
            if (hours.length != 2) {
                throw new IllegalArgumentException("Invalid operating hours format");
            }

            LocalTime startTime = LocalTime.parse(hours[0], DateTimeFormatter.ofPattern("HH:mm"));
            LocalTime endTime = LocalTime.parse(hours[1], DateTimeFormatter.ofPattern("HH:mm"));
            System.out.println("IsBefore: " + startTime.isBefore(endTime) + " and equals: " + startTime.equals(endTime));

            for (int i = 0; i < numberOfTables; i++) {
                TableEntity table = new TableEntity();

//                table.initializeTimeSlots(startTime, endTime);

                table.setCapacity(tableCapacities.get(i));
                table.setLocation(tableLocations.get(i));

                String smokingAreaString = tableSmokingAreas.get(i);
                table.setSmokingArea(smokingAreaString.equalsIgnoreCase("on"));
                table.setDescription(tableDescriptions.get(i));
                table.setRestaurant(restaurant);

                tableRepository.save(table);
            }
        } catch (DateTimeParseException e) {
            System.out.println("Error parsing operating hours: " + e.getMessage());
        } catch (IllegalArgumentException e) {
            System.out.println("Invalid operating hours format: " + e.getMessage());
        }
    }
    @Override
    public Restaurant findById(Long restaurantId) {
        return restaurantRepository.findById(restaurantId).orElseThrow(InvalidRestaurantIdException::new);
    }

    @Override
    public Restaurant updateRestaurant(Long restaurantId, String name, String cuisineType, String address, String phone, String operatingHours, String website, String socialMediaLinks, BigDecimal rating, List<Long> tablesList) {
        List<TableEntity> tableEntities = tableRepository.findAllById(tablesList);

        Restaurant restaurant = restaurantRepository.findById(restaurantId).orElseThrow(InvalidRestaurantIdException::new);
        restaurant.setName(name);
        restaurant.setCuisineType(cuisineType);
        restaurant.setAddress(address);
        restaurant.setPhone(phone);
        restaurant.setOperatingHours(operatingHours);
        restaurant.setWebsite(website);
        restaurant.setSocialMediaLinks(socialMediaLinks);
        restaurant.setRating(rating);
        restaurant.setTablesList(tableEntities);

        return restaurantRepository.save(restaurant);
    }

    @Override
    public Restaurant deleteRestaurant(Long restaurantId) {
        Restaurant restaurant = restaurantRepository.findById(restaurantId).orElseThrow(InvalidRestaurantIdException::new);
        restaurantRepository.delete(restaurant);
        return restaurant;
    }

    @Override
    public List<Restaurant> listRestaurantBy(String search) {
        // Get all restaurants from the repository
        List<Restaurant> allRestaurants = restaurantRepository.findAll();


        return allRestaurants.stream()
                .filter(restaurant ->
                        restaurant.getName().toLowerCase().contains(search.toLowerCase()) ||
                                restaurant.getCuisineType().toLowerCase().contains(search.toLowerCase())
                )
                .collect(Collectors.toList());
    }

    @Override
    public List<Restaurant> getRestaurantsWithAvailableTimeSlotsForToday() {
        LocalDate today = LocalDate.now();
        List<Restaurant> restaurants = restaurantRepository.findAll();
        List<Restaurant> restaurantsWithAvailableTimeSlots = new ArrayList<>();

        for (Restaurant restaurant : restaurants) {
            // Check if the restaurant has available time slots for today
            boolean hasAvailableTimeSlots = tableService.hasAvailableTimeSlotsForRestaurantAndDate(restaurant, today);
            if (hasAvailableTimeSlots) {
                restaurantsWithAvailableTimeSlots.add(restaurant);
            }
        }

        return restaurantsWithAvailableTimeSlots;
    }

    @Override
    public List<Restaurant> findRestaurantsByDateTimeAndPartySize(LocalDateTime dateTime, int partySize, String search) {
        List<Restaurant> allRestaurants = restaurantRepository.findAll();
        return allRestaurants.stream()
                .filter(restaurant -> hasAvailableTable(restaurant, dateTime, partySize))
                .filter(restaurant -> isMatch(restaurant, search))
                .collect(Collectors.toList());
    }

    private boolean hasAvailableTable(Restaurant restaurant, LocalDateTime dateTime, int partySize) {
        for (TableEntity table : restaurant.getTablesList()) {
            if (table.isAvailable(dateTime) && table.getCapacity() >= partySize) {
                return true;
            }
        }
        return false;
    }

    private boolean isMatch(Restaurant restaurant, String name) {
        return name == null || name.isEmpty() || restaurant.getName().contains(name);
    }

    @Override
    public List<Restaurant> findRestaurantsBySearchParams(LocalDateTime dateTime, int partySize, String search) {
        if (!search.isEmpty()) {
            List<Restaurant> restaurantList = null;
            if (!restaurantRepository.findAllByNameLike(search).isEmpty()) {
                restaurantList = restaurantRepository.findAllByNameLike(search);
            } else {
                restaurantList = restaurantRepository.findAllByCuisineTypeLike(search);
            }
            return restaurantList;
        } else {
            List<TableEntity> tableEntities = tableRepository.findAllByTimeSlotsContainingAndCapacity(dateTime, partySize);
            return tableEntities.stream()
                    .map(TableEntity::getRestaurant)
                    .distinct()  // To avoid duplicates in case one restaurant has multiple tables
                    .collect(Collectors.toList());
        }
    }
}
