import React, { useState, useEffect } from 'react';
import { useParams } from 'react-router-dom';
import axios from 'axios';
import { useNavigate } from 'react-router-dom';
import {jwtDecode} from "jwt-decode";

const ReservationConfirmation = () => {
    const navigate = useNavigate();

    const [restaurant, setRestaurant] = useState({});
    const [table, setTable] = useState({});
    const [reservationDateTime, setReservationDateTime] = useState('');
    const [partySize, setPartySize] = useState('');
    const [specialRequests, setSpecialRequests] = useState('');
    const { tableNumber, timeSlot, restaurantId } = useParams();

    useEffect(() => {
        const fetchTableDetails = async () => {
            try {
                const tableResponse = await axios.get(`http://localhost:8081/api/tables/${tableNumber}`);
                setTable(tableResponse.data);

                const restaurantResponse = await axios.get(`http://localhost:8081/api/restaurants/${restaurantId}`);
                setRestaurant(restaurantResponse.data);
            } catch (error) {
                console.error('Error fetching table details:', error);
            }
        };
        fetchTableDetails();
    }, [tableNumber, restaurantId]);

    const handleSubmit = async (e) => {
        e.preventDefault();

        if (!restaurant || !table) {
            console.error("Restaurant or table is missing.");
            return;
        }

        try {
            const token = localStorage.getItem("token");
            if (!token) {
                console.error("No token found");
                return;
            }

            // Decode the token to get the user email
            const decodedToken = jwtDecode(token);
            console.log(decodedToken)
            const userId = decodedToken.iss;// Assuming the email is part of the decoded JWT token

            const response = await axios.post(
                `http://localhost:8081/api/reservations/${userId}`,
                {
                    restaurant: restaurant,
                    table: table,
                    checkInTime: timeSlot,
                    partySize: partySize,
                    specialRequests: specialRequests
                },
                {
                    headers: {
                        Authorization: `Bearer ${token}` // Include the token here
                    }
                }
            );

            navigate("/reservations");
        } catch (error) {
            console.error("Error creating reservation:", error);
        }
    };

    const initialRemainingTime = localStorage.getItem('remainingTime') || 300;
    const [remainingTime, setRemainingTime] = useState(parseInt(initialRemainingTime));

    useEffect(() => {
        const timer = setInterval(() => {
            setRemainingTime((prevTime) => {
                const newTime = prevTime - 1;
                localStorage.setItem('remainingTime', newTime.toString()); // Update remaining time in localStorage
                return newTime;
            });
        }, 1000);

        return () => clearInterval(timer);
    }, []);

    useEffect(() => {
        // Reset remaining time if it reaches zero
        if (remainingTime <= 0) {
            localStorage.removeItem('remainingTime');
            // Optionally, handle releasing the hold on the table
        }
    }, [remainingTime]);

    const formatTime = (timeInSeconds) => {
        const minutes = Math.floor(timeInSeconds / 60);
        const seconds = timeInSeconds % 60;
        return `${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
    };

    return (
        <div className="container mt-5">
            <div className="row justify-content-center">
                <div className="col-md-6">
                    <div className="card">
                        <div className="card-header">
                            <h3 className="text-center">Reservation Confirmation</h3>
                            <p>Remaining Time: {formatTime(remainingTime)}</p>
                        </div>
                        <form onSubmit={handleSubmit}>
                            <div className="card-body">
                                <h5 className="card-title">Reservation Details</h5>
                                <p className="card-text">
                                    <strong>Restaurant:</strong> {restaurant.name || 'Loading...'} <br/>
                                    <strong>Cuisine type:</strong> {restaurant.cuisineType || 'Loading...'} <br/>
                                    <strong>Selected Time Slot:</strong> {timeSlot} <br/>
                                    <strong>Party size:</strong> <input type="number" max={table.capacity}
                                                                        value={partySize}
                                                                        onChange={(e) => setPartySize(e.target.value)}/>
                                    <strong>Table size:</strong> {table.capacity} <br/>
                                    <strong>Special Requests:</strong> <input type="text" value={specialRequests}
                                                                              onChange={(e) => setSpecialRequests(e.target.value)}/><br/>
                                </p>
                                <p className="card-text text-success">
                                    <strong>Check-in Time: grace period of 15 minutes +- the slot, for more call the
                                        restaurant</strong><br/>
                                </p>
                            </div>
                            <div className="card-footer">
                                <button type="submit" className="btn btn-primary">Submit</button>
                            </div>
                            <div className="card-footer">
                                <a href="/restaurants" className="btn btn-primary">Back to Restaurants</a>
                            </div>
                            <div className="card-footer">
                                <a href="/" className="btn btn-primary">Back to Home</a>
                            </div>
                        </form>
                    </div>
                </div>

            </div>
        </div>
    );
};

export default ReservationConfirmation;
