import React, { useState, useEffect } from 'react';
import axios from 'axios';
import {useNavigate, useParams} from 'react-router-dom';
import StarRating from "./StarRating";
import {jwtDecode} from "jwt-decode";

const ReservationEdit = () => {
    const navigate = useNavigate();

    const [isLoading, setIsLoading] = useState(true);
    const { reservationId } = useParams(); // Extract reservationId from URL params
    const [formData, setFormData] = useState({});
    const [table, setTable] = useState({});
    const [restaurant, setRestaurant] = useState({});
    const [timeSlots, setTimeSlots] = useState([]);
    const [filteredTimeSlots, setFilteredTimeSlots] = useState([]);
    const [checkInTime, setCheckInTime] = useState([]);

    useEffect(() => {
        const fetchReservation = async () => {
            try {
                setIsLoading(true);
                const response = await axios.get(`http://localhost:8081/api/reservations/${reservationId}`);
                setCheckInTime(response.data.checkInTime)
                setFormData(response.data);
                setTable(response.data.table);
                setRestaurant(response.data.restaurant);
                setTimeSlots(response.data.table.timeSlots);
                setIsLoading(false);
            } catch (error) {
                console.error('Error fetching reservation:', error);
            }
        };
        fetchReservation();
    }, [reservationId]);

    useEffect(() => {
        if (!table || !restaurant) return; // If table or restaurant is not loaded, return early

        const currentTime = new Date();

        const filteredSlots = timeSlots.filter(timeSlot => new Date(timeSlot) >= currentTime);
        setFilteredTimeSlots(filteredSlots);
    }, [table, restaurant]);

    const handleInputChange = (e) => {
        const { name, value } = e.target;

        // Check if the changed input is the time slot select element
        if (name === 'selectedTimeSlot') {
            // Update the formData with the selected time slot value
            setFormData({ ...formData, checkInTime: value });
        } else {
            // For other input fields, update formData as usual
            setFormData({ ...formData, [name]: value });
        }
    };

    const handleSubmit = async (e) => {
        e.preventDefault();

        try {
            const token = localStorage.getItem("token");
            if (!token) {
                console.error("No token found");
                return;
            }

            const decodedToken = jwtDecode(token);
            console.log(decodedToken)
            const userId = decodedToken.iss;

            await axios.post(`http://localhost:8081/api/reservations/${reservationId}/${userId}`, formData);

            navigate(`/reservations`)
        } catch (error) {
            console.error('Error updating reservation:', error);
        }
    };

    const formatTimeSlot = (timeSlot) => {
        const date = new Date(timeSlot);
        const formattedDate = date.toLocaleDateString();
        const formattedTime = date.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
        return `${formattedDate} - ${formattedTime}`;
    };

    const formatCurrentTimeSlot = (timeSlot) => {
        const date = new Date(timeSlot);
        const formattedDate = date.toLocaleDateString('en-GB'); // Format date as YYYY-MM-DD
        const formattedTime = date.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' }); // Format time as HH:MM
        return `${formattedDate} ${formattedTime}`;
    };

    return (
        <div className="container">
            {isLoading ? ( // Conditional rendering based on loading state
                <p>Loading...</p>
            ) : (
                <>
                    <h1>Edit Reservation</h1>
                    <div className="card-body">
                        <h2>
                            {restaurant.name} <StarRating key={restaurant.id} rating={restaurant.rating}/>
                        </h2>
                        <p className="card-text">{restaurant.cuisineType}</p>
                        <p className="card-text">{restaurant.operatingHours}</p>
                        <p className="card-text">Ul. {restaurant.address}</p>
                        <br/>
                    </div>
                    <form onSubmit={handleSubmit}>
                        <div className="mb-3">
                            <label htmlFor="checkInTime" className="form-label">Check-in Time</label>
                            <select className="form-select mt-2" aria-label="Select Time Slot"
                                    name="selectedTimeSlot" // Add name attribute
                                    onChange={handleInputChange}>
                                <option value="">Select Time Slot</option>
                                {filteredTimeSlots.map((timeSlot, index) => (
                                    <option key={index} value={timeSlot}>{formatTimeSlot(timeSlot)}</option>
                                ))}
                            </select>
                            <label className=".text-danger">Current check in time: {formatTimeSlot(checkInTime)}</label>
                        </div>
                        <div className="mb-3">
                            <label htmlFor="partySize" className="form-label">Party Size</label>
                            <input type="number" className="form-control" id="partySize" name="partySize"
                                   max={table.capacity}
                                   value={formData.partySize || ''} onChange={handleInputChange}/>
                        </div>
                        <div className="mb-3">
                        <label htmlFor="specialRequests" className="form-label">Special Requests</label>
                            <input type="text" className="form-control" id="specialRequests" name="specialRequests"
                                   value={formData.specialRequests || ''} onChange={handleInputChange}/>
                        </div>
                        <button type="submit" className="btn btn-primary">Submit</button>
                    </form>
                </>
            )}
        </div>
    );

};

export default ReservationEdit;
