import React, { useEffect, useState } from "react";
import {jwtDecode} from "jwt-decode";
import axios from "axios";

const ReservationHistory = () => {
    const [reservations, setReservations] = useState([]);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState(null);

    useEffect(() => {
        const fetchReservations = async () => {
            try {
                const token = localStorage.getItem("token");
                if (!token) {
                    setError("User not authenticated.");
                    setLoading(false);
                    return;
                }

                const decodedToken = jwtDecode(token);
                const userId = decodedToken.iss;

                const response = await axios.get(
                    `http://localhost:8081/api/past-reservations/${userId}`,
                    {
                        headers: { Authorization: `Bearer ${token}` },
                    }
                );

                setReservations(response.data);
            } catch (err) {
                setError("Failed to fetch reservations.");
            } finally {
                setLoading(false);
            }
        };

        fetchReservations();
    }, []);

    if (loading) return <div>Loading...</div>;
    if (error) return <div className="alert alert-danger">{error}</div>;

    return (
        <div className="container mt-5">
            <h3>Past Reservations</h3>
            <table className="table table-bordered">
                <thead className="thead-dark">
                <tr>
                    <th>#</th>
                    <th>Restaurant</th>
                    <th>Table</th>
                    <th>Date & Time</th>
                    <th>Party Size</th>
                    <th>Special Requests</th>
                    <th>Status</th>
                    <th>Cancellation Reason</th>
                </tr>
                </thead>
                <tbody>
                {reservations.map((res, index) => (
                    <tr key={res.id}>
                        <td>{index + 1}</td>
                        <td>{res.restaurant?.name || "N/A"}</td>
                        <td>{res.table?.id || "N/A"}</td>
                        <td>{new Date(res.reservationDateTime).toLocaleString()}</td>
                        <td>{res.partySize}</td>
                        <td>{res.specialRequests || "None"}</td>
                        <td>{res.status}</td>
                        <td>{res.cancellationReason || "None"}</td>
                    </tr>
                ))}
                </tbody>
            </table>
        </div>
    );
};

export default ReservationHistory;
