package com.example.rezevirajmasa.demo.service.impl;

import com.example.rezevirajmasa.demo.dto.UserDto;
import com.example.rezevirajmasa.demo.mappers.UserMapper;
import com.example.rezevirajmasa.demo.model.*;
import com.example.rezevirajmasa.demo.model.exceptions.InvalidReservationException;
import com.example.rezevirajmasa.demo.model.exceptions.InvalidReservationIdException;
import com.example.rezevirajmasa.demo.repository.CustomerRepository;
import com.example.rezevirajmasa.demo.repository.ReservationRepository;
import com.example.rezevirajmasa.demo.repository.RestaurantRepository;
import com.example.rezevirajmasa.demo.repository.TableRepository;
import com.example.rezevirajmasa.demo.service.ReservationHistoryService;
import com.example.rezevirajmasa.demo.service.ReservationService;
import com.example.rezevirajmasa.demo.service.UserService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cglib.core.Local;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.annotation.AuthenticationPrincipal;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.stereotype.Service;
import org.springframework.web.bind.annotation.RequestBody;

import javax.swing.text.html.Option;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Optional;

@Service
public class ReservationImpl implements ReservationService {
    @Autowired
    private TableRepository tableRepository;
    @Autowired
    private ReservationHistoryService reservationHistoryService;
    private final UserMapper userMapper;
    @Autowired
    private UserService userService;
    @Autowired
    private ReservationRepository reservationRepository;

    public ReservationImpl(UserMapper userMapper) {
        this.userMapper = userMapper;
    }

    @Override
    public Reservation findById(Long id) {
        return reservationRepository.findById(id).orElseThrow(InvalidReservationIdException::new);
    }

    @Override
    public void makeReservation(User user, TableEntity table, Restaurant restaurant, LocalDateTime localDateTime, LocalDateTime checkInTime, int partySize, String specialRequests) {
        if (!table.isAvailable(checkInTime)) {
            // Handle unavailability (throw an exception, return a specific response, etc.)
            throw new RuntimeException("Table not available for the specified time slot");
        }

        Reservation reservation =
                new Reservation(user, table, restaurant, LocalDateTime.now(), partySize, specialRequests, "Reserved", checkInTime, checkInTime.plusHours(2), null);

//        // Update table status or perform additional logic if needed
//        tableRepository.save(table);

        // Save the reservation
        reservationRepository.save(reservation);
    }

    @Override
    public Reservation makeReservationRest(Reservation reservation, User user) {
        Optional<TableEntity> optionalTable = tableRepository.findById(reservation.getTable().getId());
        if (optionalTable.isPresent()) {
            TableEntity table = optionalTable.get();

            reservation.setUser(user);

            LocalDateTime startTime = reservation.getCheckInTime().minusHours(2);
            LocalDateTime endTime = reservation.getCheckInTime().plusHours(2);

            table.getTimeSlots().removeIf(
                    x -> x.isAfter(startTime) && x.isBefore(endTime)
            );
            reservation.setReservationDateTime(LocalDateTime.now());
            return reservationRepository.save(reservation);
        } else {
            throw new InvalidReservationException("Unsuccessful reservation -> time slot not available");
        }
    }


    @Override
    public List<Reservation> listAll() {
        return reservationRepository.findAll();
    }

    @Override
    public Reservation getReservationById(Long reservationId) {
        return reservationRepository.findById(reservationId).orElseThrow(InvalidReservationIdException::new);
    }

    @Override
    public boolean cancelReservation(Long reservationId) {
        Optional<Reservation> optionalReservation = reservationRepository.findById(reservationId);
        if (optionalReservation.isPresent()) {
            Reservation reservation = optionalReservation.get();
            TableEntity table = reservation.getTable();

            LocalDateTime from = reservation.getCheckInTime().minusHours(2);
            LocalDateTime till = reservation.getCheckInTime().plusHours(2);

            String[] hours = table.getRestaurant().getOperatingHours().split("-");
            LocalTime openingHourTime = LocalTime.parse(hours[0], DateTimeFormatter.ofPattern("HH:mm"));
            LocalDateTime openingDateTime = openingHourTime.atDate(from.toLocalDate());
            LocalTime closingHourTime = LocalTime.of(22,45);
            LocalDateTime closingDateTime = closingHourTime.atDate(till.toLocalDate());
            if(from.isBefore(openingDateTime)) {
                from = openingDateTime;
            }
            if(till.isAfter(closingDateTime)) {
                till = closingDateTime;
            }
            while (from.isBefore(reservation.getCheckInTime().plusHours(2))) {
                table.addTimeSlot(from);
                from = from.plusMinutes(15);
            }
            reservationHistoryService.moveReservationToHistory(reservation, "Canceled", "Canceled by user");
            reservationRepository.delete(reservation);
            return true;
        } else {
            return false;
        }
    }


    @Override
    public List<Reservation> findReservationByUser(User user) {
        LocalDateTime now = LocalDateTime.now();
        return reservationRepository.findALlByUserAndCheckInTimeAfter(user, now);
    }

    @Override
    public List<Reservation> findReservationsByUserPast(User user) {
        LocalDateTime now = LocalDateTime.now();
        return reservationRepository.findALlByUserAndCheckInTimeBefore(user, now);
    }

    @Override
    public List<Reservation> findReservationsByTableAndDateRange(TableEntity table, LocalDateTime startDateTime, LocalDateTime endDateTime) {
        return reservationRepository.findByTableAndCheckInTimeBetween(table, startDateTime, endDateTime);
    }

    @Override
    public List<Reservation> findReservationsToMove(LocalDateTime currentTime) {
        return reservationRepository.findAllByCheckInTimeBefore(currentTime);
    }

    @Override
    public void deleteReservation(Long reservationID) {
        Reservation reservation = findById(reservationID);
        reservationRepository.delete(reservation);
    }
}
