package com.example.rezevirajmasa.demo.web.controller;

import com.example.rezevirajmasa.demo.dto.SignUpDto;
import com.example.rezevirajmasa.demo.dto.UserDto;
import com.example.rezevirajmasa.demo.mappers.UserMapper;
import com.example.rezevirajmasa.demo.model.*;
import com.example.rezevirajmasa.demo.model.exceptions.ExpiredReservationException;
import com.example.rezevirajmasa.demo.service.*;
import org.springframework.format.annotation.DateTimeFormat;
import org.springframework.security.core.Authentication;
import org.springframework.stereotype.Controller;
import org.springframework.transaction.annotation.RestrictedTransactionalEventListenerFactory;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.*;

import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.List;
import java.util.stream.Collectors;

@Controller
public class ReservationController {
    private final ReservationService reservationService;
    private final UserService userService;
    private final TableService tableService;
    private final ReservationHistoryService reservationHistoryService;

    private final UserMapper userMapper;

    public ReservationController(ReservationService reservationService, UserService userService, TableService tableService, ReservationHistoryService reservationHistoryService, UserMapper userMapper) {
        this.reservationService = reservationService;
        this.userService = userService;
        this.tableService = tableService;
        this.reservationHistoryService = reservationHistoryService;
        this.userMapper = userMapper;
    }

    @GetMapping("/reservations")
    public String showReservations(Model model, Authentication authentication) {
        UserDto userDto = userService.findByEmail(authentication.getName());

        SignUpDto signUpDto = userMapper.toSignUpDto(userDto);

        User user = userMapper.signUpToUser(signUpDto);

        List<Reservation> reservationList = reservationService.findReservationByUser(user);

        model.addAttribute("bodyContent", "reservationList");
        model.addAttribute("reservations", reservationList);
        return "index";
    }


    @GetMapping("/reservations/edit/{reservationId}")
    public String showEditReservation(@PathVariable Long reservationId, Model model) {
        Reservation reservation = reservationService.findById(reservationId);
        tableService.canceledTimeSlots(reservation.getTable().getId(), reservation.getCheckInTime());
        model.addAttribute("reservation", reservation);
        model.addAttribute("bodyContent", "reservationsEditForm");
        return "index";
    }

    @PostMapping("/restaurants/{tableNumber}")
    public String saveReservation(@PathVariable Long tableNumber,
                                  @RequestParam @DateTimeFormat(iso = DateTimeFormat.ISO.DATE_TIME) LocalDateTime reservationDateTime,
                                  @RequestParam int partySize,
                                  @RequestParam String specialRequests,
                                  Authentication authentication){
        TableEntity table = tableService.findById(tableNumber);
        Restaurant restaurant = table.getRestaurant();

        UserDto userDto = userService.findByEmail(authentication.getName());

        SignUpDto signUpDto = userMapper.toSignUpDto(userDto);

        User user = userMapper.signUpToUser(signUpDto);

        if(!table.isAvailable(reservationDateTime)) {
            System.out.println("The slot is taken sorry, pick another :D");
        }
        tableService.deleteTimeSlotsForReservation(tableNumber, reservationDateTime);
        reservationService.makeReservation(user, table, restaurant, LocalDateTime.now(), reservationDateTime, partySize, specialRequests);
        return "redirect:/reservations";
    }

    @RequestMapping("/reservations/edit/{reservationId}")
    public String updateReservation(@PathVariable Long reservationId,
                                    @RequestParam int partySize,
                                    @RequestParam String specialRequests,
                                    @RequestParam @DateTimeFormat(iso = DateTimeFormat.ISO.DATE_TIME) LocalDateTime reservationDateTime) {
        Reservation reservation = reservationService.findById(reservationId);
        reservation.setPartySize(partySize);
        reservation.setSpecialRequests(specialRequests);
        reservation.setCheckInTime(reservationDateTime);
        tableService.deleteTimeSlotsForReservation(reservation.getTable().getId(), reservationDateTime);
        return "redirect:/reservations";
    }

    @PostMapping("/reservations/delete/{reservationId}")
    public String deleteReservation(@PathVariable Long reservationId, @RequestParam String reason) {
        Reservation reservation = reservationService.findById(reservationId);
        tableService.canceledTimeSlots(reservation.getTable().getId(), reservation.getCheckInTime());
        reservationService.cancelReservation(reservationId);
        reservationHistoryService.moveReservationToHistory(reservation, "canceled", reason);
        return "redirect:/reservation-history";
    }
}
