package com.example.rezevirajmasa.demo.web.rest;

import com.example.rezevirajmasa.demo.dto.CustomerDTO;
import com.example.rezevirajmasa.demo.dto.UserDto;
import com.example.rezevirajmasa.demo.mappers.UserMapper;
import com.example.rezevirajmasa.demo.model.*;
import com.example.rezevirajmasa.demo.service.*;
import com.example.rezevirajmasa.demo.service.impl.TokenService;
import jakarta.servlet.http.HttpServletRequest;
import jdk.jfr.consumer.RecordingStream;
import org.apache.coyote.Response;
import org.apache.http.protocol.HTTP;
import org.springframework.format.annotation.DateTimeFormat;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.annotation.AuthenticationPrincipal;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.core.parameters.P;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.web.bind.annotation.*;

import java.nio.file.attribute.UserPrincipal;
import java.security.Principal;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

@CrossOrigin(origins = "http://localhost:3000/")
@RestController
public class testController {
    private final RestaurantService restaurantService;
    private final CustomerService customerService;
    private final UserService userService;
    private final ReservationService reservationService;
    private final ReservationHistoryService reservationHistoryService;
    private final TableService tableService;
    private final UserMapper userMapper;
    private final TokenService tokenService;
    public testController(RestaurantService restaurantService, CustomerService customerService, UserService userService, ReservationService reservationService, ReservationHistoryService reservationHistoryService, TableService tableService, UserMapper userMapper, TokenService tokenService) {
        this.restaurantService = restaurantService;
        this.customerService = customerService;
        this.userService = userService;
        this.reservationService = reservationService;
        this.reservationHistoryService = reservationHistoryService;
        this.tableService = tableService;
        this.userMapper = userMapper;
        this.tokenService = tokenService;
    }

    @GetMapping("/api/memberships")
    public ResponseEntity<List<MembershipLevel>> getMemeberships() {
        return new ResponseEntity<List<MembershipLevel>>(List.of(MembershipLevel.values()), HttpStatus.OK);
    }

    //restaurants

    @RequestMapping("/api/restaurants")
    public ResponseEntity<List<Restaurant>> getAllRestaurants() {
        return new ResponseEntity<List<Restaurant>>(restaurantService.listall(), HttpStatus.OK);
    }

    @RequestMapping("/api/restaurants/{restaurantId}")
    public ResponseEntity<Restaurant> getRestaurantById(@PathVariable Long restaurantId) {
        return new ResponseEntity<Restaurant>(restaurantService.findById(restaurantId), HttpStatus.OK);
    }

    @PostMapping("/api/search")
    public ResponseEntity<List<Restaurant>> searchRestaurants(@RequestBody Map<String, Object> requestData) {
        Optional<String> dateTimeOptional = Optional.ofNullable((String) requestData.get("dateTime"));
        int partySize = Integer.parseInt(requestData.get("partySize").toString());
        Optional<String> searchOptional = Optional.ofNullable((String) requestData.get("search"));

        String dateTime = dateTimeOptional.orElse(null);
        String search = searchOptional.orElse(null);

        LocalDateTime parsedDateTime = null;
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm");
        if (dateTime != null) {
            parsedDateTime = LocalDateTime.parse(dateTime, formatter);
        }

        List<Restaurant> filteredRestaurants = restaurantService.findRestaurantsBySearchParams(parsedDateTime, partySize, search);

        return new ResponseEntity<List<Restaurant>>(filteredRestaurants, HttpStatus.OK);
    }

    @PostMapping("/api/search/shortcut/{param}")
    public ResponseEntity<List<Restaurant>> searchByCuisineTypeShortcut(@PathVariable String param) {
        List<Restaurant> filteredRestaurants;
        if(param != null && !param.isEmpty()) {
            filteredRestaurants = restaurantService.findRestaurantsByCuisineType(param);
        } else {
            filteredRestaurants = restaurantService.listall();
        }
        return new ResponseEntity<List<Restaurant>>(filteredRestaurants, HttpStatus.OK);
    }

    @GetMapping("/api/cuisineTypes")
    public ResponseEntity<List<String>> getAllCuisineTypes() {
        List<String> cuisineTypes = restaurantService.findALlCuisineTypes();
        return new ResponseEntity<>(cuisineTypes, HttpStatus.OK);
    }

//    Customer CALLS

    @RequestMapping("/api/customers")
    public ResponseEntity<List<CustomerDTO>> getAllCustomers() {
        List<Customer> customers = customerService.listAll();
        List<CustomerDTO> dtos = new ArrayList<>();
        for(Customer customer : customers) {
            CustomerDTO dto = customerService.mapCustomerToDTO(customer);
            dtos.add(dto);
        }
        return new ResponseEntity<List<CustomerDTO>>(dtos, HttpStatus.OK);
    }

    @RequestMapping("/api/customers/{id}")
    public ResponseEntity<Customer> getCustomerById(@PathVariable Long id) {
        return new ResponseEntity<Customer>(customerService.findById(id), HttpStatus.OK);
    }
    @PutMapping("/api/customers/edit/{id}")
    public ResponseEntity<Customer> editCustomerById(@PathVariable Long id, @RequestBody Customer customer) {
        return new ResponseEntity<Customer>(
                customerService.updateCustomer(id, customer.getFirstName(), customer.getLastName(), customer.getEmail(), customer.getPassword(), customer.getPhone(), customer.getAddress(), customer.getMembershipLevel()),
                HttpStatus.OK
        );
    }
    @GetMapping("/api/user")
    public ResponseEntity<?> getCurrentUser() {
        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();

        if (authentication == null || !authentication.isAuthenticated() || authentication.getPrincipal().equals("anonymousUser")) {
            return new ResponseEntity<>("User is not authenticated", HttpStatus.UNAUTHORIZED);
        }

        Customer customer = (Customer) authentication.getPrincipal();
        return new ResponseEntity<>(customer, HttpStatus.OK);
    }


    @PostMapping("/api/customers")
    public ResponseEntity<Customer> saveCustomer(@RequestBody Customer customer) {
        // Ensure that the role is set to ROLE_USER by default
        customer.setRole(Role.ROLE_USER);
        return new ResponseEntity<Customer>(
                customerService.registration(customer),
                HttpStatus.OK
        );
    }

    @DeleteMapping("/api/customers/delete/{customerId}")
    public ResponseEntity<Response> deleteCustomer(@PathVariable Long customerId) {
        boolean deleted = customerService.deleteById(customerId);
        if(deleted) {
            return ResponseEntity.ok().build();
        } else {
            return ResponseEntity.notFound().build();
        }
    }

//    Reservation Calls

    @GetMapping("/api/reservations/by/{email}")
    public ResponseEntity<List<Reservation>> getReservations(@PathVariable String email) {
        User user = userService.findByMail(email);
        return new ResponseEntity<List<Reservation>>(reservationService.findReservationByUser(user), HttpStatus.OK);
    }

    @PostMapping("/api/reservations/{email}")
    public ResponseEntity<Reservation> createReservation(@RequestBody Reservation reservation, @PathVariable String email) {
        User user = userService.findByMail(email);
        Reservation savedReservation = reservationService.makeReservationRest(reservation, user);

        return new ResponseEntity<>(savedReservation, HttpStatus.CREATED);
    }


    @GetMapping("/api/reservations/{reservationId}")
    public ResponseEntity<Reservation> getReservation(@PathVariable Long reservationId) {
        Reservation reservation = reservationService.findById(reservationId);
        if (reservation != null) {
            return ResponseEntity.ok(reservation);
        } else {
            return ResponseEntity.notFound().build();
        }
    }

    @PostMapping("/api/reservations/{reservationId}/{email}")
    public ResponseEntity<Reservation> editReservation(@PathVariable Long reservationId,
                                                       @RequestBody Reservation reservation,
                                                       @PathVariable String email) {
        User user = userService.findByMail(email);

        if (!reservation.getReservationID().equals(reservationId)) {
            return ResponseEntity.badRequest().build();
        }

        // Fetch existing reservation
        Reservation existingReservation = reservationService.findById(reservationId);
        if (existingReservation == null) {
            return ResponseEntity.notFound().build();
        }

        if (!reservation.getCheckInTime().equals(existingReservation.getCheckInTime())) {
            tableService.canceledTimeSlots(existingReservation.getTable().getId(), existingReservation.getCheckInTime());

            tableService.deleteTimeSlotsForReservation(reservation.getTable().getId(), reservation.getCheckInTime());
        }

        existingReservation.setReservationDateTime(LocalDateTime.now());
        existingReservation.setCheckInTime(reservation.getCheckInTime());
        existingReservation.setCheckOutTime(reservation.getCheckInTime().plusHours(2));
        existingReservation.setPartySize(reservation.getPartySize());
        existingReservation.setSpecialRequests(reservation.getSpecialRequests());

        Reservation updatedReservation = reservationService.makeReservationRest(existingReservation, user);

        System.out.println("Updated reservation time: " + updatedReservation.getCheckInTime());
        return ResponseEntity.ok(updatedReservation);
    }


    @DeleteMapping("/api/reservations/delete/{reservationId}")
    public ResponseEntity<Response> deleteReservation(@PathVariable Long reservationId) {
        boolean deleted = reservationService.cancelReservation(reservationId);
        if(deleted) {
            return ResponseEntity.ok().build();
        } else {
            return ResponseEntity.notFound().build();
        }
    }

    @GetMapping("/api/reservations/past/{email}")
    public ResponseEntity<List<Restaurant.ReservationHistory>> pastReservations(@PathVariable String email) {
        User user = userService.findByMail(email);
        return null;
    }

//    TableEntity Calls

    @GetMapping("/api/tables/{tableId}")
    public ResponseEntity<TableEntity> findTableById(@PathVariable Long tableId) {
        return new ResponseEntity<TableEntity>(tableService.findById(tableId), HttpStatus.OK);
    }

    // past reservation calls
    @GetMapping("/api/past-reservations/{email}")
    public ResponseEntity<List<Restaurant.ReservationHistory>> findPastReservationsByUser(@PathVariable String email) {
        User user = userService.findByMail(email);
        List<Restaurant.ReservationHistory> reservations = reservationHistoryService.findByUser(user);
        return new ResponseEntity<>(reservations, HttpStatus.OK);
    }

}
