import {BrowserRouter as Router, Route, Routes, useNavigate} from 'react-router-dom';
import Customers from './components/Customers';
import Layout from "./components/Layout";
import React, {useContext, useEffect, useState} from 'react';
import CustomerFormContainer from "./components/CustomerFormContainer";
import CustomerDetails from "./components/CustomerDetails";
import ErrorPage from "./components/ErrorPage";
import Restaurants from "./components/Restaurants";
import Reservations from "./components/Reservations";
import RestaurantDetails from "./components/RestaurantDetails";
import ReservationConfirmation from "./components/ReservationConfirmation";
import ReservationEdit from "./components/ReservationEdit";
import axios from "axios";
import { CuisineContext } from './components/CuisineContext';
import RestaurantInfo from "./components/RestaurantInfo";
import LoginForm from "./components/Login";

const App = () => {
    return (
        <Router>
            <Layout>
                <Routes>
                    <Route path="/" element={<Home />} />
                    <Route path="/customers" element={<Customers />} />
                    <Route path="/customers/add" element={<CustomerFormContainer/>} />
                    <Route path="/customers/:id" element={<CustomerDetails />} />
                    <Route path="/customers/edit/:id" element={<CustomerFormContainer/>} />
                    <Route path="/restaurants" element={<Restaurants />} />
                    <Route path="/restaurants/:id" element={<RestaurantDetails />} />
                    <Route path="/reservations" element={<Reservations />} />
                    <Route path="/reservationConfirmation/:tableNumber/:timeSlot/:restaurantId" element={<ReservationConfirmation />} />
                    <Route path="/reservations/reservationEdit/:reservationId" element={<ReservationEdit />} />
                    <Route path={"/login"} exact render={() => <LoginForm login={this.login}/>}/>
                    <Route path="/error" element={<ErrorPage/>}/>
                </Routes>
            </Layout>
        </Router>
    );
}


const Home = () => {
    const navigate = useNavigate();

    const todayDate = new Date().toISOString().split('T')[0]; // Get today's date in 'YYYY-MM-DD' format

    const [date, setDate] = useState(todayDate);
    const [selectedTime, setSelectedTime] = useState('');
    const [numPeople, setNumPeople] = useState(2);
    const [searchValue, setSearchValue] = useState('');
    const [timeSlots, setTimeSlots] = useState([]);
    let [filteredRestaurants, setFilteredRestaurants] = useState([]);

    const cuisineTypes = useContext(CuisineContext);
    const [showCuisineSearch, setShowCuisineSearch] = useState(true);

    const [formatedDateTime, setFormatedDateTime] = useState('')

    useEffect(() => {
        if (date) {
            const selectedDate = new Date(date);
            const today = new Date();
            const isToday = selectedDate.toDateString() === today.toDateString();

            // Determine the start hour and minute
            let startHour = 9;
            let startMinute = 0;
            if (isToday) {
                const currentHour = today.getHours();
                const currentMinute = today.getMinutes();
                // If current time is later than 09:00, start from the current hour and minute
                if (currentHour > 9 || (currentHour === 9 && currentMinute >= 0)) {
                    startHour = currentHour;
                    startMinute = Math.ceil(currentMinute / 15) * 15;
                }
            }

            // Create the start time and end time
            const startTime = new Date(selectedDate.getFullYear(), selectedDate.getMonth(), selectedDate.getDate(), startHour, startMinute);
            const endTime = new Date(selectedDate.getFullYear(), selectedDate.getMonth(), selectedDate.getDate(), 23, 30);

            // Generate time slots from start time to end time in 15-minute intervals
            const slots = [];
            let currentTime = new Date(startTime);
            while (currentTime <= endTime) {
                const option = currentTime.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit', hour12: false });
                slots.push(option);
                currentTime.setMinutes(currentTime.getMinutes() + 15); // Increment by 15 minutes
            }

            // Update the timeSlots state
            setTimeSlots(slots);
        }
    }, [date]);

    const handleDateChange = (e) => {
        setDate(e.target.value);
    };

    const handleTimeChange = (e) => {
        setSelectedTime(e.target.value);
    };

    const handleNumPeopleChange = (e) => {
        setNumPeople(e.target.value);
    };

    const handleInputChange = (event) => {
        setSearchValue(event.target.value);
    };

    const handleSubmit = async (e) => {
        e.preventDefault();
        const [year, month, day] = date.split("-");
        let formattedDateTime;

        if (selectedTime) {
            const [selectedHours, selectedMinutes] = selectedTime.split(":");
            // Check if selectedHours and selectedMinutes are valid numbers
            if (!isNaN(selectedHours) && !isNaN(selectedMinutes)) {
                const dateTime = new Date(Date.UTC(year, month - 1, day, selectedHours, selectedMinutes));
                formattedDateTime = dateTime.toISOString().slice(0, 16).replace('T', ' ');
                setFormatedDateTime(formattedDateTime);
            }
        } else {
            // Find the first available time slot after the current time
            const now = new Date();
            const currentTime = now.getHours() * 60 + now.getMinutes(); // Current time in minutes
            const nextSlot = timeSlots.find(slot => {
                const [hours, minutes] = slot.split(":");
                const slotTime = parseInt(hours) * 60 + parseInt(minutes); // Time of the slot in minutes
                return slotTime > currentTime;
            });

            // If no slot is found after the current time, use the first slot of the day
            formattedDateTime = nextSlot ? `${date} ${nextSlot}` : `${date} ${timeSlots[0]}`;
        }

        const data = {
            dateTime: formattedDateTime,
            partySize: numPeople,
            search: searchValue
        };

        console.log("Data to be submitted:");
        console.log(data);

        try {
            const response = await axios.post('http://localhost:8080/api/search', data);
            const filteredRestaurants = response.data;
            setFilteredRestaurants(filteredRestaurants);
            console.log(filteredRestaurants);
            setShowCuisineSearch(false);
            // Handle response accordingly
        } catch (error) {
            console.error('Error:', error);
        }
    };

    const handleSearchByCuisine = async (cuisine) => {
        const cuisineName = cuisine.replace('Searching by cuisine: ', '');
        try {
            const response = await axios.post(`http://localhost:8080/api/search/shortcut/${cuisineName}`, cuisineName);
            console.log(response.data);
            setFilteredRestaurants(response.data)
        } catch (error) {
            console.error('Error searching by cuisine:', error);
        }
        setShowCuisineSearch(false);
    };

    const handleTimeSlotClick = (table, timeSlot, restaurant) => {
        const tableNumber = table.id;
        const formattedTimeSlot = timeSlot;
        const restaurantId = restaurant.restaurantId;

        const encodedTableNumber = encodeURI(tableNumber);
        const encodedTimeSlot = encodeURIComponent(formattedTimeSlot);
        const encodedRestaurantId = encodeURIComponent(restaurantId);

        navigate(`/reservationConfirmation/${encodedTableNumber}/${encodedTimeSlot}/${encodedRestaurantId}`);
    }

    const renderTimeSlots = (tablesList, restaurant) => {
        const [year, month, day] = date.split("-");
        const [hours, minutes] = selectedTime.split(":");
        const dateTime = new Date(year, month - 1, day, hours, minutes); // month is zero-based

        let timestamp = dateTime.getTime();
        if (isNaN(timestamp)) {
            timestamp = Date.now();
        }

        let renderedTimeSlots = {}; // Object to store rendered slots for each table capacity

        return tablesList.flatMap(table => {
            // Render capacity header when encountering a new capacity
            if (!renderedTimeSlots[table.capacity] && numPeople <= table.capacity) {
                renderedTimeSlots[table.capacity] = 0;
                return (
                    <div key={table.capacity}>
                        <h3>Table for: {table.capacity}</h3>
                        {table.timeSlots.map((timeSlot, index) => {
                            let timeSlotTime = new Date(timeSlot).getTime();

                            const tableCapacity = table.capacity;
                            // Check if the time slot is after the current time, numPeople is less than or equal to tableCapacity, and limit to 5 slots
                            if (timeSlotTime >= timestamp && numPeople <= tableCapacity && renderedTimeSlots[tableCapacity] < 5) {
                                renderedTimeSlots[tableCapacity]++;
                                const timeSlotDateTime = new Date(timeSlot);
                                const formattedDateTime = timeSlotDateTime.toLocaleString(); // Format for both date and time

                                return (
                                    <button key={index} className="btn btn-primary me-2 mb-2" onClick={() => handleTimeSlotClick(table, timeSlot, restaurant)}>
                                        {formattedDateTime} {/* Display both date and time */}
                                    </button>
                                );
                            } else {
                                return null; // Render nothing if the condition is not met
                            }
                        })}
                    </div>
                );
            } else {
                // If capacity has been rendered, return null to avoid duplicate rendering
                return null;
            }
        });
    }



// Rest of your component code...

    const today = new Date();
    const year = today.getFullYear();
    const month = String(today.getMonth() + 1).padStart(2, '0');
    const day = String(today.getDate()).padStart(2, '0');
    const formattedDate = `${year}-${month}-${day}`;


    return (
        <div className="container">
            <h2 className="display-1">Rezerviraj masa</h2>
            <form className="row g-2 align-items-center" onSubmit={handleSubmit}>
                <div className="col-auto">
                    <input className="form-control me-2" type="date" value={date} onChange={handleDateChange}
                           min={formattedDate}/>
                </div>
                <div className="col-auto">
                    <select className="form-select" onChange={handleTimeChange}>
                        {timeSlots.map((slot, index) => (
                            <option key={index} value={slot}>{slot}</option>
                        ))}
                    </select>
                </div>
                <div className="col-auto">
                    <select className="form-select" value={numPeople} onChange={handleNumPeopleChange}>
                        {[...Array(20).keys()].map((num) => (
                            <option key={num + 1} value={num + 1}>{num + 1}</option>
                        ))}
                    </select>
                </div>
                <div className="col-auto">
                    <input
                        className="form-control me-2"
                        type="search"
                        name="search"
                        placeholder="Restaurant or Cuisine"
                        aria-label="Search"
                        value={searchValue} // Set the value of the input field
                        onChange={handleInputChange} // Call the event handler on change
                    />
                </div>
                <div className="col-auto">
                    <button className="btn btn-outline-success" type="submit">Search</button>
                </div>

                <div className="border-0">
                    {filteredRestaurants.map((restaurant) => (
                        <div key={restaurant.id} className="card mb-3">
                            <div className="card-body">
                                <RestaurantInfo key={restaurant.id} restaurant={restaurant}/>
                                <p>Available time slots</p>
                                <div className="d-flex flex-wrap">
                                    {renderTimeSlots(restaurant.tablesList.flatMap((table) => table), restaurant)}
                                </div>
                            </div>
                        </div>
                    ))}
                </div>


                {showCuisineSearch && (
                    <div className="mb-3">
                        <h2 className="display-2">Search by cuisine type</h2>
                    <ul className="list-group">
                        {cuisineTypes.map((cuisine, index) => (
                            <li key={index} className="list-group-item">
                                <button type="button" className="btn btn-outline-primary"
                                        onClick={() => handleSearchByCuisine(cuisine)}>
                                    {cuisine}
                                </button>
                            </li>
                        ))}
                    </ul>
                </div>)}
            </form>
        </div>
    );
}


export default App;
