import React, {useState, useEffect, useContext} from 'react';
import axios from 'axios';
import 'bootstrap/dist/css/bootstrap.min.css';
import {useNavigate} from "react-router-dom";
import {jwtDecode} from "jwt-decode";
import {RestaurantContext} from "./RestaurantContext";


const Reservations = () => {
    const navigate = useNavigate();
    const { restaurants } = useContext(RestaurantContext);
    const [reservations, setReservations] = useState([]);

    useEffect(() => {
        const fetchReservations = async () => {
            try {
                const token = localStorage.getItem("token");
                if (!token) {
                    console.error("No token found");
                    return;
                }
                const decodedToken = jwtDecode(token);
                const userId = decodedToken.iss;

                const response = await axios.get(`http://localhost:8081/api/reservations/by/${userId}`);
                setReservations(response.data);
            } catch (error) {
                console.error('Error fetching reservations:', error);
            }
        };

        fetchReservations();
    }, []);

    const handleEditReservation = async (reservationId) => {
        if(reservationId!=null) {
            navigate(`/reservations/reservationEdit/${reservationId}`);
        }
    }

    const handleCancelReservation = async (reservationID) => {
        try {
            await axios.delete(`http://localhost:8081/api/reservations/delete/${reservationID}`);
            setReservations(reservations.filter(reservation => reservation.reservationID !== reservationID));
            alert('Reservation canceled successfully!');
        } catch (error) {
            console.error('Error canceling reservation:', error);
            alert('An error occurred while canceling the reservation. Please try again later.');
        }
    };
    return (
        <div className="container">
            <div className="row">
                {reservations.length === 0 ? (
                    <div className="text-center mt-5">
                        <h4>No active reservations</h4>
                        <p>Looking for a place to dine? Check out our <a href="/restaurants" className="text-primary">restaurants</a>.</p>
                    </div>
                ) : (
                    reservations.map(reservation => (
                        <div key={reservation.reservationID} className="col-md-4 mb-4">
                            <div className="card h-100">
                                <div className="card-body">
                                    <h5 className="card-title">Reservation ID: {reservation.reservationID}</h5>
                                    <p className="card-text">
                                        Restaurant: {
                                        restaurants.find(r => r.restaurantId === reservation.restaurantId)?.name || "Not specified"
                                    }
                                    </p>

                                    <p className="card-text">Table
                                        Number: {reservation?.tableNumber || "Not specified"}</p>
                                    <p className="card-text">
                                        Reservation Date: {reservation.checkInTime ?
                                        new Date(reservation.checkInTime).toLocaleDateString('en-US', {
                                            weekday: 'long', year: 'numeric', month: 'long', day: 'numeric'
                                        }) :
                                        "Not specified"}
                                    </p>
                                    <p className="card-text">
                                        Reservation Time: {reservation.checkInTime ?
                                        new Date(reservation.checkInTime).toLocaleTimeString('en-US', {
                                            hour: 'numeric', minute: 'numeric', hour12: true
                                        }) :
                                        "Not specified"}
                                    </p>
                                    <p className="card-text">Reservation made on: {reservation.reservationDateTime ?
                                        new Date(reservation.reservationDateTime).toLocaleTimeString('en-US', {
                                            hour: 'numeric', minute: 'numeric', hour12: true
                                        }) :
                                        "Not specified"}  {reservation.reservationDateTime ?
                                        new Date(reservation.reservationDateTime).toLocaleDateString('en-US', {
                                            weekday: 'long', year: 'numeric', month: 'long', day: 'numeric'
                                        }) :
                                        "Not specified"} </p>
                                    <p className="card-text">Party Size: {reservation.partySize || "Not specified"}</p>
                                    {reservation.preOrderedItems && reservation.preOrderedItems.length > 0 ? (
                                        <div className="mt-3">
                                            <h5 className="text-primary">Pre-Ordered Items:</h5>
                                            <ul className="list-group mb-3">
                                                {reservation.preOrderedItems.map((item, index) => (
                                                    <li key={index} className="list-group-item d-flex justify-content-between align-items-center">
                                                        <span><strong>{item?.preorderedItemName}</strong> × {item.quantity}</span>
                                                        <span className="badge bg-success rounded-pill">${(item.price * item.quantity).toFixed(2)}</span>
                                                    </li>
                                                ))}
                                            </ul>

                                            <div className="alert alert-info text-center" role="alert">
                                                <h5>Grand Total: ${reservation.preOrderedItems.reduce((acc, item) => acc + (item.quantity * item.price), 0).toFixed(2)}</h5>
                                            </div>
                                        </div>
                                    ) : (
                                        <p>No pre-ordered items.</p>
                                    )}
                                    <p className="card-text text-danger">Special Requests: {reservation.specialRequests || "None"}</p>
                                    <p className="card-text">Status: {reservation.status || "Pending"}</p>
                                    <p className="card-text text-danger">Grace period of 15 minutes +-</p>
                                    <div className="d-flex justify-content-between mt-3">
                                        <button
                                            className="btn btn-warning"
                                            onClick={() => handleEditReservation(reservation.reservationID)}>
                                            Edit Reservation
                                        </button>
                                        <button
                                            className="btn btn-danger"
                                            onClick={() => handleCancelReservation(reservation.reservationID)}>
                                            Cancel Reservation
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </div>
                    ))
                )}
            </div>
        </div>
    );
};

export default Reservations;

