import React, { useEffect, useState } from "react";
import { jwtDecode } from "jwt-decode";
import axios from "axios";
import 'bootstrap/dist/css/bootstrap.min.css';
import 'bootstrap/dist/js/bootstrap.bundle.min';

const ReservationHistory = () => {
    const [reservations, setReservations] = useState([]);
    const [filteredReservations, setFilteredReservations] = useState([]);
    const [selectedReservation, setSelectedReservation] = useState(null);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState(null);

    const [restaurantFilter, setRestaurantFilter] = useState("");
    const [tableFilter, setTableFilter] = useState("");
    const [partySizeFilter, setPartySizeFilter] = useState("");
    const [statusFilter, setStatusFilter] = useState("");
    const [cancellationReasonFilter, setCancellationReasonFilter] = useState("");
    const [startDate, setStartDate] = useState("");
    const [endDate, setEndDate] = useState("");

    useEffect(() => {
        const fetchReservations = async () => {
            try {
                const token = localStorage.getItem("token");
                if (!token) {
                    setError("User not authenticated.");
                    setLoading(false);
                    return;
                }

                const decodedToken = jwtDecode(token);
                const userId = decodedToken.iss;

                const response = await axios.get(
                    `http://localhost:8081/api/past-reservations/${userId}`,
                    {
                        headers: { Authorization: `Bearer ${token}` },
                    }
                );

                setReservations(response.data);
                setFilteredReservations(response.data);
            } catch (err) {
                setError("Failed to fetch reservations.");
            } finally {
                setLoading(false);
            }
        };

        fetchReservations();
    }, []);

    useEffect(() => {
        let tempReservations = reservations;

        if (restaurantFilter) {
            tempReservations = tempReservations.filter(res =>
                res.restaurant?.name.toLowerCase().includes(restaurantFilter.toLowerCase())
            );
        }
        if (tableFilter) {
            tempReservations = tempReservations.filter(res =>
                res.table?.id?.toString() === tableFilter
            );
        }
        if (partySizeFilter) {
            tempReservations = tempReservations.filter(res =>
                res.partySize?.toString() === partySizeFilter
            );
        }
        if (statusFilter) {
            tempReservations = tempReservations.filter(res =>
                res.status.toLowerCase().includes(statusFilter.toLowerCase())
            );
        }
        if (cancellationReasonFilter) {
            tempReservations = tempReservations.filter(res =>
                (res.cancellationReason || "None").toLowerCase().includes(cancellationReasonFilter.toLowerCase())
            );
        }
        if (startDate && endDate) {
            const start = new Date(startDate);
            const end = new Date(endDate);

            tempReservations = tempReservations.filter(res => {
                const reservationDate = new Date(res.reservationDateTime);
                return reservationDate >= start && reservationDate <= end;
            });
        }

        setFilteredReservations(tempReservations);
    }, [
        restaurantFilter,
        tableFilter,
        partySizeFilter,
        statusFilter,
        cancellationReasonFilter,
        startDate,
        endDate,
        reservations
    ]);

    const resetFilters = () => {
        setRestaurantFilter("");
        setTableFilter("");
        setPartySizeFilter("");
        setStatusFilter("");
        setCancellationReasonFilter("");
        setStartDate("");
        setEndDate("");
    };

    if (loading) return <div>Loading...</div>;
    if (error) return <div className="alert alert-danger">{error}</div>;

    const formatDateTime = (dateString) => {
        if (!dateString) return "N/A";
        const date = new Date(dateString);
        return date.toLocaleString("en-GB", {
            day: "2-digit",
            month: "short",
            year: "numeric",
            hour: "2-digit",
            minute: "2-digit",
        });
    };

    return (
        <div className="container mt-5">
            <h3 className="mb-4 text-center">Past Reservations</h3>

            <div className="row mb-4 align-items-end">
                <div className="col-md-2">
                    <input
                        type="text"
                        className="form-control"
                        placeholder="Filter by Restaurant"
                        value={restaurantFilter}
                        onChange={(e) => setRestaurantFilter(e.target.value)}
                    />
                </div>
                <div className="col-md-2">
                    <input
                        type="number"
                        className="form-control"
                        placeholder="Filter by Table ID"
                        value={tableFilter}
                        onChange={(e) => setTableFilter(e.target.value)}
                    />
                </div>
                <div className="col-md-4 d-flex gap-2">
                    <input
                        type="date"
                        className="form-control"
                        value={startDate}
                        onChange={(e) => setStartDate(e.target.value)}
                    />
                    <input
                        type="date"
                        className="form-control"
                        value={endDate}
                        onChange={(e) => setEndDate(e.target.value)}
                    />
                </div>
                <div className="col-md-2">
                    <input
                        type="number"
                        className="form-control"
                        placeholder="Party Size"
                        value={partySizeFilter}
                        onChange={(e) => setPartySizeFilter(e.target.value)}
                    />
                </div>
                <div className="col-md-2">
                    <select
                        value={statusFilter}
                        onChange={(e) => setStatusFilter(e.target.value)}
                        className="form-control"
                    >
                        <option value="">Filter by Status</option>
                        <option value="successful">Successful</option>
                        <option value="canceled">Canceled</option>
                    </select>
                </div>
                <div className="col-md-2 mt-2">
                    <input
                        type="text"
                        className="form-control"
                        placeholder="Cancellation Reason"
                        value={cancellationReasonFilter}
                        onChange={(e) => setCancellationReasonFilter(e.target.value)}
                    />
                </div>
                <div className="col-md-2 mt-2">
                    <button
                        onClick={resetFilters}
                        className="btn btn-outline-secondary w-100"
                    >
                        Reset Filters
                    </button>
                </div>
            </div>

            <div className="table-responsive">
                <table className="table table-striped table-hover table-bordered align-middle">
                    <thead className="table-dark text-center">
                    <tr>
                        <th>#</th>
                        <th>Restaurant</th>
                        <th>Table</th>
                        <th>Check In Date</th>
                        <th>Reserved on</th>
                        <th>Party Size</th>
                        <th>Special Requests</th>
                        <th>Status</th>
                        <th>Cancellation Reason</th>
                    </tr>
                    </thead>
                    <tbody>
                    {filteredReservations.length > 0 ? (
                        filteredReservations.map((res, index) => (
                            <tr key={res.id} onClick={() => setSelectedReservation(res)} style={{ cursor: "pointer" }}>
                                <td>{index + 1}</td>
                                <td>{res.restaurant?.name || "N/A"}</td>
                                <td>{res.table?.id || "N/A"}</td>
                                <td>{formatDateTime(res.checkInDate)}</td>
                                <td>{new Date(res.reservationDateTime).toLocaleString()}</td>
                                <td>{res.partySize}</td>
                                <td>{res.specialRequests || "None"}</td>
                                <td>{res.status}</td>
                                <td>{res.cancellationReason || "None"}</td>
                            </tr>
                        ))
                    ) : (
                        <tr>
                            <td colSpan="9" className="text-center">No reservations found.</td>
                        </tr>
                    )}
                    </tbody>
                </table>
            </div>

            {/* Modal for reservation details */}
            {selectedReservation && (
                <div className="modal fade show d-block" tabIndex="-1" style={{ backgroundColor: "rgba(0,0,0,0.5)" }} onClick={() => setSelectedReservation(null)}>
                    <div className="modal-dialog modal-lg" onClick={(e) => e.stopPropagation()}>
                        <div className="modal-content">
                            <div className="modal-header">
                                <h5 className="modal-title">Reservation Details</h5>
                                <button type="button" className="btn-close" onClick={() => setSelectedReservation(null)}></button>
                            </div>
                            <div className="modal-body">
                                <h6>🍽️ Restaurant Info</h6>
                                <p><strong>Name:</strong> {selectedReservation.restaurant?.name}</p>
                                <p><strong>Address:</strong> {selectedReservation.restaurant?.address}</p>
                                <p><strong>Phone:</strong> {selectedReservation.restaurant?.phone}</p>
                                <p><strong>Rating:</strong> {selectedReservation.restaurant?.rating} ⭐</p>
                                <p><strong>Cuisine:</strong> {selectedReservation.restaurant?.cuisineType}</p>
                                <p><strong>Hours:</strong> {selectedReservation.restaurant?.operatingHours}</p>
                                {selectedReservation.restaurant?.website && (
                                    <p><strong>Website:</strong> <a href={`https://${selectedReservation.restaurant.website}`} target="_blank" rel="noopener noreferrer">{selectedReservation.restaurant.website}</a></p>
                                )}

                                <hr />

                                <h6>🪑 Table Info</h6>
                                <p><strong>Location:</strong> {selectedReservation.table?.location}</p>
                                <p><strong>Capacity:</strong> {selectedReservation.table?.capacity}</p>
                                <p><strong>Smoking Area:</strong> {selectedReservation.table?.smokingArea ? "Yes" : "No"}</p>
                                <p><strong>Description:</strong> {selectedReservation.table?.description}</p>
                                <p><strong>Reservation Duration:</strong> {selectedReservation.table?.reservationDurationHours} hours</p>

                                <hr />

                                <h6>📋 Reservation Info</h6>
                                <p><strong>Check-In Date:</strong> {formatDateTime(selectedReservation.checkInDate)}</p>
                                <p><strong>Reserved on:</strong> {new Date(selectedReservation.reservationDateTime).toLocaleString()}</p>
                                <p><strong>Party Size:</strong> {selectedReservation.partySize}</p>
                                <p><strong>Special Requests:</strong> {selectedReservation.specialRequests || "None"}</p>
                                <p><strong>Status:</strong> {selectedReservation.status}</p>
                                <p><strong>Cancellation Reason:</strong> {selectedReservation.cancellationReason || "None"}</p>
                            </div>
                            <div className="modal-footer">
                                <button className="btn btn-secondary" onClick={() => setSelectedReservation(null)}>Close</button>
                            </div>
                        </div>
                    </div>
                </div>
            )}
        </div>
    );
};

export default ReservationHistory;
