import React, { useState, useEffect } from 'react';
import axios from 'axios';
import 'bootstrap/dist/css/bootstrap.min.css';

const MenuList = ({ restaurantId, onPreOrderChange, setPreOrderedItems, preOrderedItems }) => {
    const [menuItems, setMenuItems] = useState([]);
    const [preOrder, setPreOrder] = useState([]);

    const handleItemSelect = (item) => {
        setPreOrderedItems(prevItems => [...prevItems, item]);
    };

    const handleItemToggle = (item) => {
        setPreOrderedItems(prevItems => {
            const exists = prevItems.find(i => i.id === item.id);
            if (exists) {
                return prevItems.filter(i => i.id !== item.id); // remove
            } else {
                return [...prevItems, item]; // add
            }
        });
    };

    useEffect(() => {
        const fetchMenu = async () => {
            try {
                const response = await axios.get(`http://localhost:8081/api/restaurant-menu/${restaurantId}`);
                setMenuItems(response.data);
            } catch (err) {
                console.error('Failed to fetch menu:', err);
            }
        };
        if (restaurantId) {
            fetchMenu();
        }
    }, [restaurantId]);

    const handleAddToPreOrder = (item) => {
        const existingItem = preOrder.find((i) => i.menuID === item.menuID);

        let updatedPreOrder;
        if (existingItem) {
            updatedPreOrder = preOrder.map((i) =>
                i.menuID === item.menuID ? { ...i, quantity: i.quantity + 1 } : i
            );
        } else {
            updatedPreOrder = [...preOrder, { ...item, quantity: 1 }];
        }

        setPreOrder(updatedPreOrder);
        setPreOrderedItems(updatedPreOrder);
    };

    const calculateTotal = () => {
        return preOrder.reduce((acc, item) => acc + item.price * item.quantity, 0).toFixed(2);
    };

    return (
        <div className="container mt-4">
            {menuItems.length > 0 && <h3 className="text-center">Menu</h3>}
            <div className="row">
                {menuItems.map((item) => (
                    <div key={item.menuID} className="col-md-4 mb-4">
                        <div className="card h-100 shadow-sm">
                            <div className="card-body d-flex flex-column">
                                <h5 className="card-title">{item.itemName}</h5>
                                <p className="card-text">{item.description}</p>
                                <h6 className="card-subtitle mb-2 text-muted">${item.price.toFixed(2)}</h6>
                                <button
                                    className="btn btn-outline-success mt-auto"
                                    onClick={() => handleAddToPreOrder(item)}
                                >
                                    Add to Pre-Order
                                </button>
                            </div>
                        </div>
                    </div>
                ))}
            </div>

            {preOrder.length > 0 && (
                <div className="mt-5">
                    <h4>Pre-Order Summary</h4>
                    <ul className="list-group">
                        {preOrder.map((item) => (
                            <li key={item.menuID} className="list-group-item d-flex justify-content-between align-items-center">
                                {item.itemName} (x{item.quantity})
                                <span>${(item.price * item.quantity).toFixed(2)}</span>
                            </li>
                        ))}
                    </ul>
                    <h5 className="mt-3">Total: ${calculateTotal()}</h5>
                </div>
            )}
        </div>
    );
};

export default MenuList;
