import React, {useState, useEffect, useContext} from 'react';
import 'bootstrap/dist/css/bootstrap.min.css';
import StarRating from './StarRating';
import { useNavigate } from 'react-router-dom';
import {RestaurantContext} from "./RestaurantContext";

const Restaurants = () => {
    const [restaurants, setRestaurants] = useState([]);
    const navigate = useNavigate();
    const restaurantContext = useContext(RestaurantContext);

    useEffect(() => {
        setRestaurants(restaurantContext.restaurants);
    }, [restaurantContext]);

    const handleDetailClick = (restaurantId) => {
        navigate(`/restaurants/${restaurantId}`);
    }

    const handleTimeSlotClick = (table, timeSlot, restaurant) => {
        const tableNumber = table.id;
        const formattedTimeSlot = timeSlot;
        const restaurantId = restaurant.restaurantId;

        const encodedTableNumber = encodeURI(tableNumber);
        const encodedTimeSlot = encodeURIComponent(formattedTimeSlot);
        const encodedRestaurantId = encodeURIComponent(restaurantId);

        navigate(`/reservationConfirmation/${encodedTableNumber}/${encodedTimeSlot}/${encodedRestaurantId}`);
    }

    const renderTimeSlots = (tablesList, restaurant) => {
        const currentTime = new Date().getTime();
        let renderedTimeSlots = {}; // Object to store rendered slots for each table capacity

        return tablesList.flatMap(table => {
            // Render capacity header when encountering a new capacity
            if (!renderedTimeSlots[table.capacity]) {
                renderedTimeSlots[table.capacity] = 0;
                return (
                    <div key={table.capacity}>
                        <h3>Table for: {table.capacity}</h3>
                        {table.timeSlots.map((timeSlot, index) => {
                            const timeSlotTime = new Date(timeSlot).getTime();
                            const tableCapacity = table.capacity;

                            // Check if the time slot is after the current time and limit to 3 slots
                            if (timeSlotTime > currentTime && renderedTimeSlots[tableCapacity] < 3) {
                                renderedTimeSlots[tableCapacity]++;
                                const timeSlotDateTime = new Date(timeSlot);
                                const formattedTime = timeSlotDateTime.toLocaleTimeString();
                                const formattedDateTime = timeSlotDateTime.toLocaleString(); // Format for both date and time

                                return (
                                    <button key={index} className="btn btn-primary me-2 mb-2" onClick={() => handleTimeSlotClick(table, timeSlot, restaurant)}>
                                        {formattedDateTime} {/* Display both date and time */}
                                    </button>
                                );
                            } else {
                                return null; // Render nothing if the condition is not met
                            }
                        })}
                    </div>
                );
            } else {
                // If capacity has been rendered, return null to avoid duplicate rendering
                return null;
            }
        });

    }


    return (
        <div className="container">
            <h2>Restaurants</h2>
            <div className="row">
                {restaurants.map((restaurant) => (
                    <div key={restaurant.id} className="col-md-4 mb-4">
                        <div className="card">
                            <div className="card-body">
                                <h5 className="card-title">
                                    {restaurant.name} <StarRating key={restaurant.id} rating={restaurant.rating}/>
                                </h5>
                                <p className="card-text">{restaurant.cuisineType}</p>
                                <p className="card-text">{restaurant.operatingHours}</p>
                                <p className="card-text">Ul. {restaurant.address}</p>
                                <div className="d-flex flex-wrap">
                                    {renderTimeSlots(restaurant.tablesList.flatMap((table) => table), restaurant)}
                                </div>
                            </div>
                            <button onClick={() => handleDetailClick(restaurant.restaurantId)}
                                    className="btn btn-primary">View Details
                            </button>
                        </div>
                    </div>
                ))}
            </div>
        </div>
    );
};

export default Restaurants;