package com.example.rezevirajmasa.demo.bootstrap;

import com.example.rezevirajmasa.demo.model.Reservation;
import com.example.rezevirajmasa.demo.model.Restaurant;
import com.example.rezevirajmasa.demo.model.TableEntity;
import com.example.rezevirajmasa.demo.repository.ReservationRepository;
import com.example.rezevirajmasa.demo.repository.TableRepository;
import jakarta.annotation.PostConstruct;
import jakarta.annotation.PreDestroy;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cglib.core.Local;
import org.springframework.stereotype.Component;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

@Component
public class TimeSlotManagementService  {
    @Autowired
    private TableRepository tableRepository;
    @Autowired
    private ReservationRepository reservationRepository;

    @PostConstruct
    public void initializeTimeSlots() {
        LocalDate startDate = LocalDate.now();
        initializeTimeSlotsForNextDays(startDate);
    }

    private void initializeTimeSlotsForNextDays(LocalDate startDate) {
        LocalDate currentDate = startDate;
        for (int i = 0; i < 3; i++) {
            Iterable<TableEntity> tableEntities = tableRepository.findAll();
            for (TableEntity tableEntity : tableEntities) {
                initializeTimeSlotsForTable(tableEntity, currentDate);
                tableRepository.save(tableEntity);
            }
            currentDate = currentDate.plusDays(1);
        }
    }

    private void initializeTimeSlotsForTable(TableEntity tableEntity, LocalDate date) {
        // Get existing time slots for the table
        List<LocalDateTime> existingTimeSlots = tableEntity.getTimeSlots().stream()
                .filter(slot -> slot.toLocalDate().isEqual(date)) // Filter time slots for the specified date
                .toList();

        // If existing time slots are not null and not empty, return without initializing new ones
        if (!existingTimeSlots.isEmpty()) {
            return;
        }

        Restaurant restaurant = tableEntity.getRestaurant();
        String[] hours = restaurant.getOperatingHours().split("-");
        LocalTime startTime = LocalTime.parse(hours[0], DateTimeFormatter.ofPattern("HH:mm"));
        LocalTime endTime = LocalTime.parse(hours[1], DateTimeFormatter.ofPattern("HH:mm"));

        List<LocalDateTime> reservationCheckIns = getReservationCheckInsAfterNow(date);

        tableEntity.initializeTimeSlots(startTime, endTime, Collections.singletonList(date), reservationCheckIns);
    }

    private List<LocalDateTime> getReservationCheckInsAfterNow(LocalDate date) {
        LocalDateTime currentTime = LocalDateTime.now();
        List<Reservation> reservations = reservationRepository.findByCheckInTimeAfterAndCheckInTimeBefore(
                LocalDateTime.of(date, LocalTime.MIN), LocalDateTime.of(date, LocalTime.MAX));
        List<LocalDateTime> checkIns = new ArrayList<>();
        for (Reservation reservation : reservations) {
            if (reservation.getCheckInTime().isAfter(currentTime)) {
                checkIns.add(reservation.getCheckInTime());
            }
        }
        return checkIns;
    }
}