package com.example.rezevirajmasa.demo.model;

import com.fasterxml.jackson.annotation.JsonBackReference;
import jakarta.persistence.*;
import org.springframework.web.bind.annotation.ModelAttribute;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.ArrayList;

import java.util.Iterator;
import java.util.List;
import java.util.stream.Collectors;

@Entity
@Table(name = "tables")
public class TableEntity {
    @Id
    @GeneratedValue(strategy = GenerationType.IDENTITY)
    @Column(name = "TableID")
    private Long id;

    @JsonBackReference
    @ManyToOne(fetch = FetchType.EAGER)
    @JoinColumn(name = "RestaurantID")
    private Restaurant restaurant;

    @Column(name = "Capacity")
    private int capacity;

    @Column(name = "Location")
    private String location;

    @Column(name = "IsSmokingArea")
    private boolean isSmokingArea;

    @Column(name = "Description", length = 1000)
    private String description;

    @ElementCollection(fetch = FetchType.EAGER)
    @CollectionTable(name = "table_time_slots", joinColumns = @JoinColumn(name = "table_id"))
    @Column(name = "time_slot", columnDefinition = "timestamp without time zone")
    @OrderBy("time_slot ASC")
    private List<LocalDateTime> timeSlots = new ArrayList<>();

    @Column(name = "reservation_duration_hours", nullable = true)
    private int reservationDurationHours = 2;
    public void cleanUnusedTimeSlots(LocalDate threeDaysAgo) {
        // Iterate over the time slots and remove those that are in the past
        timeSlots.removeIf(timeSlot -> timeSlot.isBefore(threeDaysAgo.atStartOfDay()));
    }

    public void addTimeSlot(LocalDateTime timeSlot) {
        this.timeSlots.add(timeSlot);
    }


    @Transient
    public boolean isAvailable(LocalDateTime desiredTimeSlot) {
        if (timeSlots == null || timeSlots.isEmpty()) {
            return true;
        }

        for (LocalDateTime reservedTimeSlot : timeSlots) {
            LocalDateTime endTime = reservedTimeSlot.plusHours(reservationDurationHours);

            // Check if the desired time slot overlaps with any existing reservation
            if ((desiredTimeSlot.equals(reservedTimeSlot) || desiredTimeSlot.isAfter(reservedTimeSlot))
                    && desiredTimeSlot.isBefore(endTime)) {
                return false; // Table is reserved for the desired time slot
            }
        }

        return true; // No conflicting reservations, table is available
    }

    public boolean hasTimeSlot(LocalDateTime dateTime) {
        // Check if the given dateTime exists in the list of time slots
        return timeSlots.contains(dateTime);
    }

    public List<LocalDateTime> initializeTimeSlots(LocalTime startTime, LocalTime endTime, List<LocalDate> dates, List<LocalDateTime> listOfUsedCheckIns) {
        // Check if time slots are already initialized and cover all provided dates
        List<LocalDate> generatedDates = timeSlots.stream()
                .map(LocalDateTime::toLocalDate)
                .distinct()
                .toList();

        if (generatedDates.containsAll(dates) && dates.containsAll(generatedDates)) {
            System.out.println("Time slots already cover all provided dates.");
            return timeSlots;
        }

        // Generate time slots for the remaining dates
        for (LocalDate date : dates) {
            if (!generatedDates.contains(date)) {
                LocalTime currentTime = startTime;
                while (currentTime.isBefore(LocalTime.of(23, 0))) {
                    LocalDateTime currentDateTime = LocalDateTime.of(date, currentTime);
                    boolean isUsed = false;
                    if (listOfUsedCheckIns != null) {
                        for (LocalDateTime checkIns : listOfUsedCheckIns) {
                            if (checkIns.isEqual(currentDateTime)) {
                                isUsed = true;
                                break;
                            }
                        }
                    }
                    if (!isUsed) {
                        timeSlots.add(currentDateTime);
                        System.out.println("Added time slot: " + currentDateTime); // Debug output
                    } else {
                        // Remove conflicting time slots
                        timeSlots.removeIf(x -> x.isAfter(currentDateTime.minusHours(2)) && x.isBefore(currentDateTime.plusHours(2)));
                    }
                    // Increment currentTime
                    currentTime = currentTime.plusMinutes(15);
                }
            }
        }

        return timeSlots;
    }



    public TableEntity(Restaurant restaurant, int capacity, String location, boolean isSmokingArea, String description, List<LocalDateTime> timeSlots, int reservationDurationHours) {
        this.restaurant = restaurant;
        this.capacity = capacity;
        this.location = location;
        this.isSmokingArea = isSmokingArea;
        this.description = description;
        this.timeSlots = timeSlots;
        this.reservationDurationHours = reservationDurationHours;
    }

    public TableEntity() {
    }

    public Long getId() {
        return id;
    }

    public void setId(Long id) {
        this.id = id;
    }

    public Restaurant getRestaurant() {
        return restaurant;
    }

    public void setRestaurant(Restaurant restaurant) {
        this.restaurant = restaurant;
    }

    public int getCapacity() {
        return capacity;
    }

    public void setCapacity(int capacity) {
        this.capacity = capacity;
    }

    public String getLocation() {
        return location;
    }

    public void setLocation(String location) {
        this.location = location;
    }

    public void setSmokingArea(Boolean isSmokingArea) {
        this.isSmokingArea = isSmokingArea;
    }

    public void setSmokingArea(boolean smokingArea) {
        isSmokingArea = smokingArea;
    }

    public String getDescription() {
        return description;
    }

    public void setDescription(String description) {
        this.description = description;
    }

    public List<LocalDateTime> getTimeSlots() {
        return timeSlots;
    }

    public void setTimeSlots(List<LocalDateTime> timeSlots) {
        this.timeSlots = timeSlots;
    }

    public int getReservationDurationHours() {
        return reservationDurationHours;
    }

    public void setReservationDurationHours(int reservationDurationHours) {
        this.reservationDurationHours = reservationDurationHours;
    }
}
