package com.example.rezevirajmasa.demo.service.impl;

import com.example.rezevirajmasa.demo.model.Customer;
import com.example.rezevirajmasa.demo.model.MembershipLevel;
import com.example.rezevirajmasa.demo.model.Reservation;
import com.example.rezevirajmasa.demo.model.Role;
import com.example.rezevirajmasa.demo.model.exceptions.InvalidCustomerIdException;
import com.example.rezevirajmasa.demo.repository.CustomerRepository;
import com.example.rezevirajmasa.demo.service.CustomerService;
import org.springframework.security.core.parameters.P;
import org.springframework.security.core.userdetails.User;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UserDetailsService;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.stereotype.Service;

import javax.swing.text.html.Option;
import java.time.Instant;
import java.time.LocalDateTime;
import java.util.*;

@Service
public class CustomerServiceImpl implements CustomerService, UserDetailsService {
    private final CustomerRepository customerRepository;

    private final PasswordEncoder passwordEncoder;
    public CustomerServiceImpl(CustomerRepository customerRepository, PasswordEncoder passwordEncoder) {
        this.customerRepository = customerRepository;
        this.passwordEncoder = passwordEncoder;
    }

    @Override
    public Customer registration(String firstName, String lastName, String email, String password, String phone, String address, MembershipLevel membershipLevel, Role role) {
        Customer customer = new Customer();
        customer.setFirstName(firstName);
        customer.setLastName(lastName);
        customer.setEmail(email);
        customer.setPassword(passwordEncoder.encode(password));
        customer.setAddress(address);
        customer.setPhone(phone);
        customer.setMembershipLevel(membershipLevel);
        customer.setRole(role);
        customer.setRegistrationDate(Date.from(Instant.now()));
        return customerRepository.save(customer);
    }

    @Override
    public List<Customer> listAll() {
        return customerRepository.findAll();
    }

    @Override
    public UserDetails loadUserByUsername(String email) throws UsernameNotFoundException {
        Customer customer = customerRepository.findByEmail(email);
        if (customer == null) {
            throw new UsernameNotFoundException("User not found with email: " + email);
        }
        return new org.springframework.security.core.userdetails.User(
                customer.getEmail(),
                customer.getPassword(),
                Collections.singletonList(customer.getRole())
        );
    }

    @Override
    public Customer updateCustomer(Long id, String firstName, String lastName, String email, String password, String phone, String address, MembershipLevel membershipLevel) {
        Customer customer = findById(id);
        customer.setFirstName(firstName);
        customer.setLastName(lastName);
        customer.setEmail(email);
        customer.setPassword(passwordEncoder.encode(password));
        customer.setPhone(phone);
        customer.setMembershipLevel(membershipLevel);
        customer.setAddress(address);
        return customerRepository.save(customer);
    }

    @Override
    public Customer findByEmail(String email) {
        return customerRepository.findByEmail(email);
    }

    @Override
    public Customer findById(Long id) {
        return customerRepository.findById(id).orElseThrow(InvalidCustomerIdException::new);
    }

    @Override
    public Customer registration(Customer customer) {
        return customerRepository.save(customer);
    }

    @Override
    public boolean deleteById(Long customerId) {
        Optional<Customer> optionalCustomer = customerRepository.findById(customerId);
        if(optionalCustomer.isPresent()) {
            Customer customer = optionalCustomer.get();
            customerRepository.delete(customer);
            return true;
        } else {
            return false;
        }
    }
}
