package com.example.rezevirajmasa.demo.service.impl;

import com.example.rezevirajmasa.demo.model.Customer;
import com.example.rezevirajmasa.demo.model.Reservation;
import com.example.rezevirajmasa.demo.model.Restaurant;
import com.example.rezevirajmasa.demo.model.TableEntity;
import com.example.rezevirajmasa.demo.model.exceptions.InvalidReservationException;
import com.example.rezevirajmasa.demo.model.exceptions.InvalidReservationIdException;
import com.example.rezevirajmasa.demo.repository.CustomerRepository;
import com.example.rezevirajmasa.demo.repository.ReservationRepository;
import com.example.rezevirajmasa.demo.repository.RestaurantRepository;
import com.example.rezevirajmasa.demo.repository.TableRepository;
import com.example.rezevirajmasa.demo.service.ReservationService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.swing.text.html.Option;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Optional;

@Service
public class ReservationImpl implements ReservationService {
    @Autowired
    private TableRepository tableRepository;
    @Autowired
    private RestaurantRepository restaurantRepository;
    @Autowired
    private CustomerRepository customerRepository;
    @Autowired
    private ReservationRepository reservationRepository;

    @Override
    public Reservation findById(Long id) {
        return reservationRepository.findById(id).orElseThrow(InvalidReservationIdException::new);
    }

    @Override
    public void makeReservation(Customer customer, TableEntity table, Restaurant restaurant, LocalDateTime localDateTime, LocalDateTime checkInTime, int partySize, String specialRequests) {
        if (!table.isAvailable(checkInTime)) {
            // Handle unavailability (throw an exception, return a specific response, etc.)
            throw new RuntimeException("Table not available for the specified time slot");
        }

        Reservation reservation =
                new Reservation(customer, table, restaurant, LocalDateTime.now(), partySize, specialRequests, "Reserved", checkInTime, checkInTime.plusHours(2), null);

//        // Update table status or perform additional logic if needed
//        tableRepository.save(table);

        // Save the reservation
        reservationRepository.save(reservation);
    }

    @Override
    public Reservation makeReservationRest(Reservation reservation) {
        Optional<TableEntity> optionalTable = tableRepository.findById(reservation.getTable().getId());
        if(optionalTable.isPresent()) {
            TableEntity table = optionalTable.get();

            LocalDateTime startTime = reservation.getCheckInTime().minusHours(2);
            LocalDateTime endTime = reservation.getCheckInTime().plusHours(2);

            table.getTimeSlots().removeIf(
                    x -> x.isAfter(startTime) && x.isBefore(endTime)
            );

            return reservationRepository.save(reservation);
        } else {
            throw new InvalidReservationException("Unsuccessful reservation -> time slot not avalaible");
        }
    }

    @Override
    public List<Reservation> listAll() {
        return reservationRepository.findAll();
    }

    @Override
    public Reservation getReservationById(Long reservationId) {
        return reservationRepository.findById(reservationId).orElseThrow(InvalidReservationIdException::new);
    }

    @Override
    public boolean cancelReservation(Long reservationId) {
        Optional<Reservation> optionalReservation = reservationRepository.findById(reservationId);
        if (optionalReservation.isPresent()) {
            Reservation reservation = optionalReservation.get();
            TableEntity table = reservation.getTable();

            LocalDateTime from = reservation.getCheckInTime().minusHours(2);
            LocalDateTime till = reservation.getCheckInTime().plusHours(2);

            String[] hours = table.getRestaurant().getOperatingHours().split("-");
            LocalTime openingHourTime = LocalTime.parse(hours[0], DateTimeFormatter.ofPattern("HH:mm"));
            LocalDateTime openingDateTime = openingHourTime.atDate(from.toLocalDate());
            LocalTime closingHourTime = LocalTime.of(22,45);
            LocalDateTime closingDateTime = closingHourTime.atDate(till.toLocalDate());
            if(from.isBefore(openingDateTime)) {
                from = openingDateTime;
            }
            if(till.isAfter(closingDateTime)) {
                till = closingDateTime;
            }
            while (from.isBefore(reservation.getCheckInTime().plusHours(2))) {
                table.addTimeSlot(from);
                from = from.plusMinutes(15);
            }
            reservationRepository.delete(reservation);
            return true; // Return true indicating successful cancellation
        } else {
            return false; // Return false indicating reservation with the given ID not found
        }
    }


    @Override
    public List<Reservation> findReservationByCustomer(Customer customer) {
        return reservationRepository.findAllByCustomer(customer);
    }

    @Override
    public List<Reservation> findReservationsByTableAndDateRange(TableEntity table, LocalDateTime startDateTime, LocalDateTime endDateTime) {
        return reservationRepository.findByTableAndCheckInTimeBetween(table, startDateTime, endDateTime);
    }

    @Override
    public List<Reservation> findReservationsToMove(LocalDateTime currentTime) {
        return reservationRepository.findAllByCheckInTimeBefore(currentTime);
    }

    @Override
    public void deleteReservation(Long reservationID) {
        Reservation reservation = findById(reservationID);
        reservationRepository.delete(reservation);
    }
}
