package com.example.rezevirajmasa.demo.service.impl;

import com.example.rezevirajmasa.demo.model.Restaurant;
import com.example.rezevirajmasa.demo.model.TableEntity;
import com.example.rezevirajmasa.demo.model.exceptions.InvalidRestaurantIdException;
import com.example.rezevirajmasa.demo.model.exceptions.InvalidTableNumberException;
import com.example.rezevirajmasa.demo.repository.RestaurantRepository;
import com.example.rezevirajmasa.demo.repository.TableRepository;
import com.example.rezevirajmasa.demo.service.TableService;
import org.springframework.cglib.core.Local;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.DateTimeException;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.chrono.ChronoLocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Iterator;
import java.util.List;

@Service
public class TableServiceImpl implements TableService {
    private final TableRepository tableRepository;
    private final RestaurantRepository restaurantRepository;

    public TableServiceImpl(TableRepository tableRepository, RestaurantRepository restaurantRepository) {
        this.tableRepository = tableRepository;
        this.restaurantRepository = restaurantRepository;
    }

    @Override
    public TableEntity findById(Long id) {
        return tableRepository.findById(id).orElseThrow(InvalidTableNumberException::new);
    }

    @Override
    public List<TableEntity> listall() {
        return tableRepository.findAll();
    }

    public void save(int numberOfTables, List<Integer> tableCapacities, List<String> tableLocations, List<String> tableSmokingAreas, List<String> tableDescriptions, Restaurant restaurant) {
        for (int i = 0; i < numberOfTables; i++) {
            TableEntity table = new TableEntity();
            table.setCapacity(tableCapacities.get(i));
            table.setLocation(tableLocations.get(i));
            table.setSmokingArea(Boolean.valueOf(tableSmokingAreas.get(i)));
            table.setDescription(tableDescriptions.get(i));
            table.setRestaurant(restaurant);
            tableRepository.save(table);
        }
    }

    @Override
    public void deleteTimeSlotsForReservation(Long tableId, LocalDateTime reservationTime) {
        LocalDateTime startTime = reservationTime.minusHours(2);
        LocalDateTime endTime = reservationTime.plusHours(2);

        TableEntity table = findById(tableId);
        List<LocalDateTime> timeSlots = table.getTimeSlots();

        timeSlots.removeIf(timeSlot -> timeSlot.isAfter(startTime) && timeSlot.isBefore(endTime));

        table.setTimeSlots(timeSlots);

        tableRepository.saveAndFlush(table);
    }

    @Override
    public void canceledTimeSlots(Long tableId, LocalDateTime reservationTime) {
        TableEntity table = findById(tableId);
        List<LocalDateTime> timeSlots = table.getTimeSlots();
        LocalDateTime startTime = reservationTime.minusHours(1).minusMinutes(45);
        LocalDateTime endTime = reservationTime.plusHours(1).plusMinutes(45);

        LocalDate localDate = reservationTime.toLocalDate();

        String[] hours = table.getRestaurant().getOperatingHours().split("-");
        LocalTime openingHourTime = LocalTime.parse(hours[0], DateTimeFormatter.ofPattern("HH:mm"));

        LocalDateTime openingHour = openingHourTime.atDate(localDate);
        LocalDateTime closingHour = LocalTime.of(23, 45).atDate(localDate);
        if(startTime.isBefore(openingHour)) {
            startTime = LocalDateTime.from(openingHour);
        }
        while(startTime.isBefore(endTime) || startTime.equals(endTime)) {
            timeSlots.add(startTime);
            startTime = startTime.plusMinutes(15);
            if(startTime.isAfter(closingHour) || startTime.equals(closingHour)) {
                break;
            }
        }
        table.setTimeSlots(timeSlots);
        tableRepository.saveAndFlush(table);
    }

    @Override
    public void saveTable(TableEntity table) {
        tableRepository.save(table);
    }

    @Override
    public TableEntity getTableByNumber(Long number) {
        return tableRepository.findById(number).orElseThrow(InvalidTableNumberException::new);
    }
    @Override
    public TableEntity deleteTable(Long number) {
        TableEntity tableEntity = tableRepository.findById(number).orElseThrow(InvalidTableNumberException::new);
        tableRepository.deleteById(number);
        return tableEntity;
    }

    @Override
    public boolean hasAvailableTimeSlotsForRestaurantAndDate(Restaurant restaurant, LocalDate today) {
        List<TableEntity> tables = tableRepository.findByRestaurant(restaurant);

        // Iterate through each table to check for available time slots
        for (TableEntity table : tables) {
            boolean hasAvailableTimeSlots = hasAvailableTimeSlotsForTableAndDate(table, today);
            if (hasAvailableTimeSlots) {
                return true; // If any table has available time slots, return true
            }
        }

        return false;
    }

    public boolean hasAvailableTimeSlotsForTableAndDate(TableEntity table, LocalDate date) {
        List<LocalDateTime> timeSlots = table.getTimeSlots();

        // Implement your logic to check if the table has available time slots for the given date
        // This could involve querying the database for reservations for the given table and date,
        // and checking if there are any open time slots.
        // You may need to consider factors such as operating hours, existing reservations, etc.
        // Return true if there are available time slots, false otherwise.
        return false;
    }

}
