package com.example.rezevirajmasa.demo.service.impl;

import com.example.rezevirajmasa.demo.dto.ReservationDTO;
import com.example.rezevirajmasa.demo.dto.RestaurantDTO;
import com.example.rezevirajmasa.demo.dto.TableDTO;
import com.example.rezevirajmasa.demo.model.Reservation;
import com.example.rezevirajmasa.demo.model.Restaurant;
import com.example.rezevirajmasa.demo.model.TableEntity;
import com.example.rezevirajmasa.demo.model.User;
import com.example.rezevirajmasa.demo.model.exceptions.InvalidRestaurantIdException;
import com.example.rezevirajmasa.demo.repository.RestaurantRepository;
import com.example.rezevirajmasa.demo.repository.TableRepository;
import com.example.rezevirajmasa.demo.service.ReservationService;
import com.example.rezevirajmasa.demo.service.RestaurantService;
import com.example.rezevirajmasa.demo.service.TableService;
import com.sun.tools.jconsole.JConsoleContext;
import jakarta.transaction.Transactional;
import org.openqa.selenium.InvalidArgumentException;
import org.springframework.stereotype.Service;
import org.modelmapper.ModelMapper;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.math.BigDecimal;
import java.time.format.DateTimeParseException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;


@Service
@Transactional
public class RestaurantServiceImpl implements RestaurantService {
    private final RestaurantRepository restaurantRepository;
    private final TableRepository tableRepository;
    private final TableService tableService;
    private final ReservationService reservationService;
    private ModelMapper modelMapper = new ModelMapper();

    public RestaurantServiceImpl(RestaurantRepository restaurantRepository, TableRepository tableRepository, TableService tableService, ReservationService reservationService) {
        this.restaurantRepository = restaurantRepository;
        this.tableRepository = tableRepository;
        this.tableService = tableService;
        this.reservationService = reservationService;
    }

    @Override
    public List<RestaurantDTO> listall() {
        List<Restaurant> restaurants = restaurantRepository.findAll();

        List<RestaurantDTO> restaurantDTOS = new ArrayList<>();

        for (Restaurant restaurant : restaurants) {
            RestaurantDTO restaurantDTO = modelMapper.map(restaurant, RestaurantDTO.class);
            List<TableDTO> tableDTOS = new ArrayList<>();
            for (TableEntity table : restaurant.getTablesList()) {
                TableDTO tableDTO = modelMapper.map(table, TableDTO.class);
                List<ReservationDTO> reservationDTOS = new ArrayList<>();
                for (Reservation reservation : table.getReservations()) {
                    ReservationDTO reservationDTO = modelMapper.map(reservation, ReservationDTO.class);

                    reservationDTOS.add(reservationDTO);
                }
                tableDTO.setReservations(reservationDTOS);
                tableDTOS.add(tableDTO);
            }
            restaurantDTO.setTablesList(tableDTOS);
            restaurantDTOS.add(restaurantDTO);
        }
        return restaurantDTOS;
    }

    @Override
    public List<Restaurant> listAll() {
        return restaurantRepository.findAll();
    }

//    @Override
//    public void save(String name, String cuisineType, String address, String phone, String operatingHours, String website, String socialMediaLinks, BigDecimal rating, List<Long> tablesList) {
//        List<TableEntity> tableEntities = tableRepository.findAllById(tablesList);
//        restaurantRepository.save(new Restaurant(name, cuisineType, address, phone, operatingHours, website, socialMediaLinks, rating, tableEntities));
//    }

    @Override
    public void save(Restaurant restaurant, int numberOfTables, List<Integer> tableCapacities, List<String> tableLocations, List<String> tableSmokingAreas, List<String> tableDescriptions) {
        if (numberOfTables != tableCapacities.size() || numberOfTables != tableLocations.size() || numberOfTables != tableSmokingAreas.size() || numberOfTables != tableDescriptions.size()) {
            throw new IllegalArgumentException("Mismatched table data. Number of tables does not match the size of the input lists.");
        }

        restaurantRepository.save(restaurant);
        String[] hours = restaurant.getOperatingHours().split("-");
        try {
            if (hours.length != 2) {
                throw new IllegalArgumentException("Invalid operating hours format");
            }

            LocalTime startTime = LocalTime.parse(hours[0], DateTimeFormatter.ofPattern("HH:mm"));
            LocalTime endTime = LocalTime.parse(hours[1], DateTimeFormatter.ofPattern("HH:mm"));

            for (int i = 0; i < numberOfTables; i++) {
                TableEntity table = new TableEntity();
                table.setCapacity(tableCapacities.get(i));
                table.setLocation(tableLocations.get(i));
                table.setSmokingArea(tableSmokingAreas.get(i).equalsIgnoreCase("on"));
                table.setDescription(tableDescriptions.get(i));
                table.setRestaurant(restaurant);
                tableRepository.save(table);
            }
        } catch (DateTimeParseException e) {
            System.out.println("Error parsing operating hours: " + e.getMessage());
        } catch (IllegalArgumentException e) {
            System.out.println("Invalid operating hours format: " + e.getMessage());
        }
    }
    @Override
    public RestaurantDTO findById(Long restaurantId) {
        Restaurant restaurant = restaurantRepository.findById(restaurantId)
                .orElseThrow(InvalidRestaurantIdException::new);

        RestaurantDTO restaurantDTO = modelMapper.map(restaurant, RestaurantDTO.class);

        List<TableDTO> tableDTOS = new ArrayList<>();
        for (TableEntity table : restaurant.getTablesList()) {
            TableDTO tableDTO = modelMapper.map(table, TableDTO.class);

            List<ReservationDTO> reservationDTOS = new ArrayList<>();
            for (Reservation reservation : table.getReservations()) {
                ReservationDTO reservationDTO = modelMapper.map(reservation, ReservationDTO.class);
                reservationDTOS.add(reservationDTO);
            }

            tableDTO.setReservations(reservationDTOS);
            tableDTOS.add(tableDTO);
        }

        restaurantDTO.setTablesList(tableDTOS);

        return restaurantDTO;
    }

    @Override
    public Restaurant findByIdRestaurant(Long restaurantId) {
        return restaurantRepository.findById(restaurantId).orElseThrow(()-> new InvalidArgumentException("No restaurant with provided id"));
    }


    @Override
    public Restaurant updateRestaurant(Long restaurantId, String name, String cuisineType, String address, String phone, String operatingHours, String website, String socialMediaLinks, BigDecimal rating, List<Long> tablesList) {
        List<TableEntity> tableEntities = tableRepository.findAllById(tablesList);

        Restaurant restaurant = restaurantRepository.findById(restaurantId).orElseThrow(InvalidRestaurantIdException::new);
        restaurant.setName(name);
        restaurant.setCuisineType(cuisineType);
        restaurant.setAddress(address);
        restaurant.setPhone(phone);
        restaurant.setOperatingHours(operatingHours);
        restaurant.setWebsite(website);
        restaurant.setSocialMediaLinks(socialMediaLinks);
        restaurant.setRating(rating);
        restaurant.setTablesList(tableEntities);

        return restaurantRepository.save(restaurant);
    }

    @Override
    public Restaurant deleteRestaurant(Long restaurantId) {
        Restaurant restaurant = restaurantRepository.findById(restaurantId).orElseThrow(InvalidRestaurantIdException::new);
        restaurantRepository.delete(restaurant);
        return restaurant;
    }

    @Override
    public List<Restaurant> listRestaurantBy(String search) {
        List<Restaurant> allRestaurants = restaurantRepository.findAll();


        return allRestaurants.stream()
                .filter(restaurant ->
                        restaurant.getName().toLowerCase().contains(search.toLowerCase()) ||
                                restaurant.getCuisineType().toLowerCase().contains(search.toLowerCase())
                )
                .collect(Collectors.toList());
    }

    @Override
    public List<Restaurant> getRestaurantsWithAvailableTimeSlotsForToday() {
        LocalDate today = LocalDate.now();
        List<Restaurant> restaurants = restaurantRepository.findAll();
        List<Restaurant> restaurantsWithAvailableTimeSlots = new ArrayList<>();

        for (Restaurant restaurant : restaurants) {
            boolean hasAvailableTimeSlots = tableService.hasAvailableTimeSlotsForRestaurantAndDate(restaurant, today);
            if (hasAvailableTimeSlots) {
                restaurantsWithAvailableTimeSlots.add(restaurant);
            }
        }

        return restaurantsWithAvailableTimeSlots;
    }

    @Override
    public List<Restaurant> findRestaurantsByDateTimeAndPartySize(LocalDateTime dateTime, int partySize, String search) {
        LocalDateTime checkOutTime = dateTime.plusHours(2);
        List<Restaurant> allRestaurants = restaurantRepository.findAll();

        return allRestaurants.stream()
                .filter(restaurant -> restaurant.getTablesList().stream()
                        .anyMatch(table -> tableRepository.findAvailableTables(dateTime, checkOutTime, partySize)
                                .contains(table)))
                .filter(restaurant -> isMatch(restaurant, search))
                .collect(Collectors.toList());
    }

    private boolean isMatch(Restaurant restaurant, String name) {
        return name == null || name.isEmpty() || restaurant.getName().contains(name);
    }

    @Override
    public List<RestaurantDTO> findRestaurantsBySearchParams(LocalDateTime dateTime, int partySize, String search) {
        List<Restaurant> availableRestaurants = new ArrayList<>();

        List<Restaurant> restaurantList;
        if (search == null || search.isEmpty()) {
            restaurantList = restaurantRepository.findAll();
        } else {
            String searchTerm = "%" + search + "%";
            restaurantList = restaurantRepository.findAllByNameLikeIgnoreCase(searchTerm);

            if (restaurantList.isEmpty()) {
                restaurantList = restaurantRepository.findAllByCuisineTypeContaining(search);
            }
        }

        for (Restaurant restaurant : restaurantList) {
            boolean hasAvailableTable = restaurant.getTablesList().stream()
                    .anyMatch(table -> table.isAvailable(dateTime, partySize));

            if (hasAvailableTable) {
                availableRestaurants.add(restaurant);
            }
        }

        List<RestaurantDTO> restaurantDTOS = availableRestaurants.stream().map(this::convertToDto).toList();
        return restaurantDTOS;
    }

    @Override
    public List<String> findALlCuisineTypes() {
        return restaurantRepository.findAllCuisineTypes();
    }

    @Override
    public List<RestaurantDTO> findRestaurantsByCuisineType(String param) {
        List<Restaurant> restaurants = restaurantRepository.findAllByCuisineType(param);
        return restaurants.stream().map(this::convertToDto).toList();
    }

    public RestaurantDTO convertToDto(Restaurant restaurant) {
        return modelMapper.map(restaurant, RestaurantDTO.class);
    }

    public TableDTO convertToDto(TableEntity table) {
        return modelMapper.map(table, TableDTO.class);
    }

    public ReservationDTO convertToDto(Reservation reservation) {
        return modelMapper.map(reservation, ReservationDTO.class);
    }
}
