package com.example.rezevirajmasa.demo.service.impl;

import com.example.rezevirajmasa.demo.dto.ReservationDTO;
import com.example.rezevirajmasa.demo.dto.RestaurantDTO;
import com.example.rezevirajmasa.demo.dto.TableDTO;
import com.example.rezevirajmasa.demo.model.Reservation;
import com.example.rezevirajmasa.demo.model.Restaurant;
import com.example.rezevirajmasa.demo.model.TableEntity;
import com.example.rezevirajmasa.demo.model.exceptions.InvalidTableNumberException;
import com.example.rezevirajmasa.demo.repository.ReservationRepository;
import com.example.rezevirajmasa.demo.repository.RestaurantRepository;
import com.example.rezevirajmasa.demo.repository.TableRepository;
import com.example.rezevirajmasa.demo.service.TableService;
import jakarta.persistence.Table;
import org.modelmapper.ModelMapper;
import org.openqa.selenium.InvalidArgumentException;
import org.springframework.cglib.core.Local;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.DateTimeException;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.chrono.ChronoLocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

@Service
public class TableServiceImpl implements TableService {
    private final TableRepository tableRepository;
    private final RestaurantRepository restaurantRepository;
    private final ReservationRepository reservationRepository;
    private ModelMapper modelMapper = new ModelMapper();

    public TableServiceImpl(TableRepository tableRepository, RestaurantRepository restaurantRepository, ReservationRepository reservationRepository) {
        this.tableRepository = tableRepository;
        this.restaurantRepository = restaurantRepository;
        this.reservationRepository = reservationRepository;
    }

    @Override
    public List<TableEntity> listall() {
        return tableRepository.findAll();
    }

    @Override
    public TableDTO findById(Long id) {
        TableEntity table = tableRepository.findById(id)
                .orElseThrow(InvalidTableNumberException::new);

        TableDTO tableDTO = modelMapper.map(table, TableDTO.class);

        List<ReservationDTO> reservationDTOS = new ArrayList<>();
        for (Reservation reservation : table.getReservations()) {
            ReservationDTO reservationDTO = modelMapper.map(reservation, ReservationDTO.class);
            reservationDTOS.add(reservationDTO);
        }

        tableDTO.setReservations(reservationDTOS);

        return tableDTO;
    }

    @Override
    public TableEntity findByIdTable(Long id) {
        return tableRepository.findById(id).orElseThrow(() -> new InvalidArgumentException("Invalid table id"));
    }


    @Override
    public void save(int numberOfTables, List<Integer> tableCapacities, List<String> tableLocations, List<String> tableSmokingAreas, List<String> tableDescriptions, Restaurant restaurant) {
        for (int i = 0; i < numberOfTables; i++) {
            TableEntity table = new TableEntity();
            table.setCapacity(tableCapacities.get(i));
            table.setLocation(tableLocations.get(i));
            table.setSmokingArea(Boolean.valueOf(tableSmokingAreas.get(i)));
            table.setDescription(tableDescriptions.get(i));
            table.setRestaurant(restaurant);
            tableRepository.save(table);
        }
    }

    @Override
    public void deleteTimeSlotsForReservation(Long tableId, LocalDateTime reservationTime) {
        TableDTO tableDTO = findById(tableId);
        TableEntity table = convertFromDTO(tableDTO);
        LocalDate threeDaysAgo = LocalDate.now().minusDays(3);
        table.cleanUnusedTimeSlots(threeDaysAgo);

        tableRepository.saveAndFlush(table);
    }

    @Override
    public void canceledTimeSlots(Long tableId, LocalDateTime reservationTime) {
        TableDTO tableDTO = findById(tableId);
        TableEntity table = convertFromDTO(tableDTO);
        LocalDateTime startTime = reservationTime.minusHours(1).minusMinutes(45);
        LocalDateTime endTime = reservationTime.plusHours(1).plusMinutes(45);

        String[] hours = table.getRestaurant().getOperatingHours().split("-");
        LocalTime openingHourTime = LocalTime.parse(hours[0], DateTimeFormatter.ofPattern("HH:mm"));
        LocalDateTime openingHour = openingHourTime.atDate(reservationTime.toLocalDate());
        LocalDateTime closingHour = LocalTime.of(23, 45).atDate(reservationTime.toLocalDate());

        if (startTime.isBefore(openingHour)) {
            startTime = openingHour;
        }

        while (startTime.isBefore(endTime) || startTime.equals(endTime)) {
            if (startTime.isAfter(closingHour) || startTime.equals(closingHour)) {
                break;
            }

            Reservation reservation = new Reservation();
            reservation.setReservationDateTime(startTime);
            reservation.setTable(table);
            reservationRepository.save(reservation);

            startTime = startTime.plusMinutes(15);
        }

        tableRepository.saveAndFlush(table);
    }

    @Override
    public void saveTable(TableEntity table) {
        tableRepository.save(table);
    }

    @Override
    public TableEntity getTableByNumber(Long number) {
        return tableRepository.findById(number).orElseThrow(InvalidTableNumberException::new);
    }

    @Override
    public TableEntity deleteTable(Long number) {
        TableEntity tableEntity = tableRepository.findById(number).orElseThrow(InvalidTableNumberException::new);
        tableRepository.deleteById(number);
        return tableEntity;
    }

    @Override
    public boolean hasAvailableTimeSlotsForRestaurantAndDate(Restaurant restaurant, LocalDate today) {
        List<TableEntity> tables = tableRepository.findByRestaurant(restaurant);

        for (TableEntity table : tables) {
            boolean hasAvailableTimeSlots = hasAvailableTimeSlotsForTableAndDate(table, today);
            if (hasAvailableTimeSlots) {
                return true;
            }
        }

        return false;
    }

    public boolean hasAvailableTimeSlotsForTableAndDate(TableEntity table, LocalDate date) {
        for (Reservation reservation : table.getReservations()) {
            LocalDateTime startTime = reservation.getReservationDateTime();
            LocalDateTime endTime = startTime.plusHours(reservation.getTable().getReservationDurationHours());

            if (startTime.toLocalDate().equals(date) || endTime.toLocalDate().equals(date)) {
                return false;
            }
        }
        return true;
    }

    public RestaurantDTO convertToDto(Restaurant restaurant) {
        return modelMapper.map(restaurant, RestaurantDTO.class);
    }

    public TableDTO convertToDto(TableEntity table) {
        return modelMapper.map(table, TableDTO.class);
    }

    public ReservationDTO convertToDto(Reservation reservation) {
        return modelMapper.map(reservation, ReservationDTO.class);
    }

    public TableEntity convertFromDTO(TableDTO tableDTO) {
        return modelMapper.map(tableDTO, TableEntity.class);
    }
}
