package com.example.rezevirajmasa.demo.web.rest;

import com.example.rezevirajmasa.demo.dto.*;
import com.example.rezevirajmasa.demo.mappers.UserMapper;
import com.example.rezevirajmasa.demo.model.*;
import com.example.rezevirajmasa.demo.model.exceptions.InvalidReservationException;
import com.example.rezevirajmasa.demo.service.*;
import com.example.rezevirajmasa.demo.service.impl.TokenService;
import jakarta.servlet.http.HttpServletRequest;
import jdk.jfr.consumer.RecordingStream;
import org.apache.coyote.Response;
import org.apache.http.protocol.HTTP;
import org.springframework.format.annotation.DateTimeFormat;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.annotation.AuthenticationPrincipal;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.core.parameters.P;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.web.bind.annotation.*;

import java.nio.file.attribute.UserPrincipal;
import java.security.Principal;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.stream.Collectors;

@CrossOrigin(origins = "http://localhost:3000/")
@RestController
public class testController {
    private final RestaurantService restaurantService;
    private final CustomerService customerService;
    private final UserService userService;
    private final ReservationService reservationService;
    private final ReservationHistoryService reservationHistoryService;
    private final TableService tableService;
    private final MenuService menuService;
    private final UserMapper userMapper;
    private final TokenService tokenService;
    public testController(RestaurantService restaurantService, CustomerService customerService, UserService userService, ReservationService reservationService, ReservationHistoryService reservationHistoryService, TableService tableService, MenuService menuService, UserMapper userMapper, TokenService tokenService) {
        this.restaurantService = restaurantService;
        this.customerService = customerService;
        this.userService = userService;
        this.reservationService = reservationService;
        this.reservationHistoryService = reservationHistoryService;
        this.tableService = tableService;
        this.menuService = menuService;
        this.userMapper = userMapper;
        this.tokenService = tokenService;
    }

    @GetMapping("/api/memberships")
    public ResponseEntity<List<MembershipLevel>> getMemeberships() {
        return new ResponseEntity<List<MembershipLevel>>(List.of(MembershipLevel.values()), HttpStatus.OK);
    }

    //restaurants calls

    @GetMapping("/api/restaurants")
    public ResponseEntity<List<RestaurantDTO>> getAllRestaurants() {
            return new ResponseEntity<List<RestaurantDTO>>(restaurantService.listall(), HttpStatus.OK);
    }

    @GetMapping("/api/restaurants/{restaurantId}")
    public ResponseEntity<RestaurantDTO> getRestaurantById(@PathVariable Long restaurantId) {
        return new ResponseEntity<RestaurantDTO>(restaurantService.findById(restaurantId), HttpStatus.OK);
    }

    @PostMapping("/api/search")
    public ResponseEntity<List<RestaurantDTO>> searchRestaurants(@RequestBody Map<String, Object> requestData) {
        Optional<String> dateTimeOptional = Optional.ofNullable((String) requestData.get("dateTime"));
        int partySize = Integer.parseInt(requestData.get("partySize").toString());
        Optional<String> searchOptional = Optional.ofNullable((String) requestData.get("search"));

        String dateTime = dateTimeOptional.orElse(null);
        String search = searchOptional.orElse(null);

        LocalDateTime parsedDateTime = null;
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm");
        if (dateTime != null) {
            parsedDateTime = LocalDateTime.parse(dateTime, formatter);
        }

        List<RestaurantDTO> filteredRestaurants = restaurantService.findRestaurantsBySearchParams(parsedDateTime, partySize, search);

        return new ResponseEntity<>(filteredRestaurants, HttpStatus.OK);
    }

    @PostMapping("/api/search/shortcut/{param}")
    public ResponseEntity<List<RestaurantDTO>> searchByCuisineTypeShortcut(@PathVariable String param) {
        List<RestaurantDTO> filteredRestaurants;
        if(param != null && !param.isEmpty()) {
            filteredRestaurants = restaurantService.findRestaurantsByCuisineType(param);
        } else {
            filteredRestaurants = restaurantService.listall();
        }
        return new ResponseEntity<List<RestaurantDTO>>(filteredRestaurants, HttpStatus.OK);
    }

    @GetMapping("/api/cuisineTypes")
    public ResponseEntity<List<String>> getAllCuisineTypes() {
        List<String> cuisineTypes = restaurantService.findALlCuisineTypes();
        return new ResponseEntity<>(cuisineTypes, HttpStatus.OK);
    }

    // User calls

    @GetMapping("/api/user/{email}")
    public ResponseEntity<User> getUserByEmail(@PathVariable String email)
    {
        User user = userService.findByMail(email);
        return ResponseEntity.ok(user);
    }

//    Customer CALLS

    @GetMapping("/api/customers")
    public ResponseEntity<List<CustomerDTO>> getAllCustomers() {
        List<Customer> customers = customerService.listAll();
        List<CustomerDTO> dtos = new ArrayList<>();
        for(Customer customer : customers) {
            CustomerDTO dto = customerService.mapCustomerToDTO(customer);
            dtos.add(dto);
        }
        return new ResponseEntity<List<CustomerDTO>>(dtos, HttpStatus.OK);
    }

    @GetMapping("/api/customers/{id}")
    public ResponseEntity<Customer> getCustomerById(@PathVariable Long id) {
        return new ResponseEntity<Customer>(customerService.findById(id), HttpStatus.OK);
    }
    @PutMapping("/api/customers/edit/{id}")
    public ResponseEntity<Customer> editCustomerById(@PathVariable Long id, @RequestBody Customer customer) {
        return new ResponseEntity<Customer>(
                customerService.updateCustomer(id, customer.getFirstName(), customer.getLastName(), customer.getEmail(), customer.getPassword(), customer.getPhone(), customer.getAddress(), customer.getMembershipLevel()),
                HttpStatus.OK
        );
    }

    @PostMapping("/api/customers")
    public ResponseEntity<Customer> saveCustomer(@RequestBody Customer customer) {
        // Ensure that the role is set to ROLE_USER by default
        customer.setRole(Role.ROLE_USER);
        return new ResponseEntity<Customer>(
                customerService.registration(customer),
                HttpStatus.OK
        );
    }

    @DeleteMapping("/api/customers/delete/{customerId}")
    public ResponseEntity<Response> deleteCustomer(@PathVariable Long customerId) {
        boolean deleted = customerService.deleteById(customerId);
        if(deleted) {
            return ResponseEntity.ok().build();
        } else {
            return ResponseEntity.notFound().build();
        }
    }

//    Reservation Calls

    @GetMapping("/api/reservations/by/{email}")
    public ResponseEntity<List<ReservationDTO>> getReservations(@PathVariable String email) {
        if (email == null || email.isEmpty()) {
            return ResponseEntity.badRequest().build();
        }
        User user = userService.findByMail(email);
        if (user == null) {
            return ResponseEntity.notFound().build();
        }

        List<Reservation> reservations = reservationService.findReservationByUser(user);
        List<ReservationDTO> reservationDTOs = reservations.stream()
                .map(ReservationDTO::new)
                .collect(Collectors.toList());

        return ResponseEntity.ok(reservationDTOs);
    }

    @PostMapping("/api/reservations")
    public ResponseEntity<?> createReservation(@RequestBody ReservationDTO reservation) {
        User user = userService.findByMail(reservation.getUserEmail());
        Reservation savedReservation = reservationService.makeReservationRest(reservation, user);

        return new ResponseEntity<>(savedReservation, HttpStatus.CREATED);
    }

    @GetMapping("/api/reservations/{reservationId}")
    public ResponseEntity<ReservationDTO> getReservation(@PathVariable Long reservationId) {
        Reservation reservation = reservationService.findById(reservationId);
        if (reservation != null) {
            return ResponseEntity.ok(new ReservationDTO(reservation));
        } else {
            return ResponseEntity.notFound().build();
        }
    }

//    @GetMapping("/api/reservations/{restaurantId}")
//    public ResponseEntity<List<Reservation>> getReservationsByRestaurant(@PathVariable Long restaurantId)
//    {
//        Restaurant restaurant = restaurantService.findByIdRestaurant(restaurantId);
//        List<Reservation> reservations = reservationService.findAllByRestaurant(restaurant);
//        return ResponseEntity.ok(reservations);
//    }

    @GetMapping("/api/table-reservations/{tableId}")
    public ResponseEntity<List<LocalDateTime>> tableReservations(@PathVariable Long tableId) {
        TableEntity table = tableService.findByIdTable(tableId);
        if (table == null) {
            return ResponseEntity.notFound().build();
        }

        List<Reservation> reservations = reservationService.reservationsForTable(table);
        List<LocalDateTime> reservedTimes = reservations.stream()
                .map(Reservation::getReservationDateTime)
                .collect(Collectors.toList());

        return ResponseEntity.ok(reservedTimes);
    }


    @PostMapping("/api/reservations/{reservationId}/{email}")
    public ResponseEntity<Reservation> editReservation(@PathVariable Long reservationId,
                                                       @RequestBody ReservationDTO reservationDTO,
                                                       @PathVariable String email) {
        User user = userService.findByMail(email);

        if (!reservationDTO.getReservationID().equals(reservationId)) {
            return ResponseEntity.badRequest().build();
        }

        try {
            Reservation updatedReservation = reservationService.updateReservation(reservationId, reservationDTO, user);
            return ResponseEntity.ok(updatedReservation);
        } catch (InvalidReservationException e) {
            return ResponseEntity.status(409).body(null);
        } catch (Exception e) {
            return ResponseEntity.status(500).build();
        }
    }



    @DeleteMapping("/api/reservations/delete/{reservationId}")
    public ResponseEntity<Response> deleteReservation(@PathVariable Long reservationId) {
        boolean deleted = reservationService.cancelReservation(reservationId);
        if(deleted) {
            return ResponseEntity.ok().build();
        } else {
            return ResponseEntity.notFound().build();
        }
    }

    @GetMapping("/api/reservations/past/{email}")
    public ResponseEntity<List<Restaurant.ReservationHistory>> pastReservations(@PathVariable String email) {
        User user = userService.findByMail(email);
        return null;
    }

    // TableEntity Calls
    @GetMapping("/api/tables/{tableId}")
    public ResponseEntity<TableDTO> findTableById(@PathVariable Long tableId) {
        TableDTO tableDTO = tableService.findById(tableId);
        return new ResponseEntity<>(tableDTO, HttpStatus.OK);
    }

    // past reservation calls
    @GetMapping("/api/past-reservations/{email}")
    public ResponseEntity<List<Restaurant.ReservationHistory>> findPastReservationsByUser(@PathVariable String email) {
        User user = userService.findByMail(email);
        List<Restaurant.ReservationHistory> reservations = reservationHistoryService.findByUser(user);
        return new ResponseEntity<>(reservations, HttpStatus.OK);
    }

    // menu calls
    @GetMapping("/api/restaurant-menu/{restaurantId}")
    public List<Menu> getMenuByRestaurantId(@PathVariable Long restaurantId) {
        return menuService.getMenuByRestaurantId(restaurantId);
    }
}
