import React, { useState, useEffect } from 'react';
import axios from 'axios';
import 'bootstrap/dist/css/bootstrap.min.css';

const ITEMS_PER_PAGE = 6;

const ReadOnlyMenuList = ({ restaurantId }) => {
    const [menuItems, setMenuItems] = useState([]);
    const [currentPage, setCurrentPage] = useState(1);

    useEffect(() => {
        const fetchMenu = async () => {
            try {
                const response = await axios.get(`http://localhost:8081/api/restaurant-menu/${restaurantId}`);
                setMenuItems(response.data);
                setCurrentPage(1);
            } catch (err) {
                console.error('Failed to fetch menu:', err);
            }
        };

        if (restaurantId) {
            fetchMenu();
        }
    }, [restaurantId]);

    const indexOfLastItem = currentPage * ITEMS_PER_PAGE;
    const indexOfFirstItem = indexOfLastItem - ITEMS_PER_PAGE;
    const currentItems = menuItems.slice(indexOfFirstItem, indexOfLastItem);
    const totalPages = Math.ceil(menuItems.length / ITEMS_PER_PAGE);

    const handlePageChange = (pageNumber) => {
        setCurrentPage(pageNumber);
    };

    return (
        <div className="container mt-4">
            {menuItems.length > 0 && <h3 className="text-center">Menu</h3>}
            <div className="row">
                {currentItems.map((item) => (
                    <div key={item.menuID} className="col-md-6 mb-4">
                        <div className="card h-100 shadow-sm">
                            <div className="card-body d-flex flex-column">
                                <h5 className="card-title">{item.itemName}</h5>
                                <p className="card-text">{item.description}</p>
                                <h6 className="card-subtitle mb-2 text-muted">${item.price.toFixed(2)}</h6>

                                {item.tags && item.tags.length > 0 && (
                                    <div className="mb-2">
                                        {item.tags.map((tag) => (
                                            <span key={tag.id} className="badge bg-secondary me-1">
                                        {tag.tagName}: {tag.tagValue}
                                    </span>
                                        ))}
                                    </div>
                                )}
                            </div>
                        </div>
                    </div>
                ))}
            </div>

            {totalPages > 1 && (
                <nav className="d-flex justify-content-center">
                    <ul className="pagination">
                        {Array.from({ length: totalPages }, (_, index) => (
                            <li
                                key={index + 1}
                                className={`page-item ${currentPage === index + 1 ? 'active' : ''}`}
                            >
                                <button
                                    type="button"
                                    className="page-link"
                                    onClick={() => handlePageChange(index + 1)}
                                >
                                    {index + 1}
                                </button>
                            </li>
                        ))}
                    </ul>
                </nav>
            )}
        </div>
    );
};

export default ReadOnlyMenuList;
