package com.example.rezevirajmasa.demo.service.impl;

import com.example.rezevirajmasa.demo.dto.PreorderedItemDto;
import com.example.rezevirajmasa.demo.dto.ReservationDTO;
import com.example.rezevirajmasa.demo.mappers.UserMapper;
import com.example.rezevirajmasa.demo.model.*;
import com.example.rezevirajmasa.demo.model.exceptions.InvalidReservationException;
import com.example.rezevirajmasa.demo.model.exceptions.InvalidReservationIdException;
import com.example.rezevirajmasa.demo.repository.ReservationRepository;
import com.example.rezevirajmasa.demo.repository.TableRepository;
import com.example.rezevirajmasa.demo.service.MenuService;
import com.example.rezevirajmasa.demo.service.ReservationHistoryService;
import com.example.rezevirajmasa.demo.service.ReservationService;
import com.example.rezevirajmasa.demo.service.UserService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

@Service
public class ReservationImpl implements ReservationService {

    @Autowired
    private TableRepository tableRepository;
    @Autowired
    private ReservationHistoryService reservationHistoryService;
    private final UserMapper userMapper;
    @Autowired
    private UserService userService;
    @Autowired
    private ReservationRepository reservationRepository;
    @Autowired
    @Lazy
    private MenuService menuService;

    public ReservationImpl(UserMapper userMapper) {
        this.userMapper = userMapper;
    }

    @Override
    public Reservation findById(Long id) {
        return reservationRepository.findById(id).orElseThrow(InvalidReservationIdException::new);
    }

    @Override
    public Reservation makeReservationRest(ReservationDTO reservationDTO, User user, Restaurant restaurant) {
        Reservation reservation = new Reservation();

        Optional<TableEntity> optionalTable = tableRepository.findById(reservationDTO.getTableNumber());

        if (optionalTable.isPresent()) {
            TableEntity table = optionalTable.get();

            LocalDateTime startTime = reservationDTO.getReservationDateTime().plusHours(1);
            LocalDateTime endTime = reservationDTO.getReservationDateTime().plusHours(3);

            boolean hasConflict = table.getReservations().stream()
                    .anyMatch(existingReservation -> {
                        LocalDateTime existingStartTime = existingReservation.getCheckInTime();
                        LocalDateTime existingEndTime = existingReservation.getCheckOutTime();

                        boolean scenario1 = existingStartTime.isBefore(startTime) && existingEndTime.isAfter(startTime);

                        boolean scenario2 = existingStartTime.isBefore(endTime) && existingEndTime.isAfter(startTime);

                        boolean scenario3 = existingStartTime.isAfter(startTime) && existingEndTime.isBefore(endTime);

                        boolean scenario4 = existingStartTime.isBefore(startTime) && existingEndTime.isAfter(endTime);

                        return scenario1 || scenario2 || scenario3 || scenario4;
                    });
            if (hasConflict) {
                throw new InvalidReservationException("Unsuccessful reservation -> time slot not available");
            }

            reservation.setTable(table);
            reservation.setReservationDateTime(LocalDateTime.now());
            reservation.setSpecialRequests(reservationDTO.getSpecialRequests());
            reservation.setPartySize(reservationDTO.getPartySize());
            reservation.setReservationStatus(reservationDTO.getStatus() != null ? reservationDTO.getStatus() : "Pending");
            reservation.setPaymentStatus(reservationDTO.getPaymentStatus() != null ? reservationDTO.getPaymentStatus() : "Unpaid");
            reservation.setUser(user);

            List<PreorderedItem> preOrderedItems = new ArrayList<>();

            for (PreorderedItemDto dtoItem : reservationDTO.getPreOrderedItems()) {
                PreorderedItem item = new PreorderedItem();
                item.setPreorderedItemName(dtoItem.getPreorderedItemName());
                item.setQuantity(dtoItem.getQuantity());
                item.setPrice(dtoItem.getPrice());
                item.setReservation(reservation);

                Menu menu = menuService.getMenuById(dtoItem.getMenuID());
                item.setMenu(menu);

                preOrderedItems.add(item);
            }

            reservation.setPreOrderedItems(preOrderedItems);
            reservation.setRestaurant(restaurant);
            reservation.setCheckInTime(reservationDTO.getReservationDateTime().plusHours(1));
            reservation.setReservationDateTime(LocalDateTime.now());
            reservation.setCheckOutTime(reservationDTO.getReservationDateTime().plusHours(3));

            return reservationRepository.save(reservation);
        } else {
            throw new InvalidReservationException("Invalid table number -> table not found");
        }
    }

    @Override
    public Reservation updateReservation(Long reservationId, ReservationDTO reservationDTO, User user) {
        Reservation existingReservation = findById(reservationId);

        if (!existingReservation.getCheckInTime().equals(reservationDTO.getReservationDateTime())) {
            TableEntity table = existingReservation.getTable();
            table.removeReservation(existingReservation);

            LocalDateTime newStartTime = reservationDTO.getReservationDateTime().minusHours(2);
            LocalDateTime newEndTime = reservationDTO.getReservationDateTime().plusHours(2);
            boolean hasConflict = table.getReservations().stream()
                    .anyMatch(r -> r.getCheckInTime().isAfter(newStartTime) && r.getCheckInTime().isBefore(newEndTime));

            if (hasConflict) {
                throw new InvalidReservationException("New time slot is not available.");
            }

            table.addReservation(existingReservation);
        }

        existingReservation.setCheckInTime(reservationDTO.getReservationDateTime());
        existingReservation.setCheckOutTime(reservationDTO.getReservationDateTime().plusHours(2));
        existingReservation.setReservationDateTime(LocalDateTime.now());
        existingReservation.setPartySize(reservationDTO.getPartySize());
        existingReservation.setSpecialRequests(reservationDTO.getSpecialRequests());
        existingReservation.setReservationStatus(reservationDTO.getStatus() != null ? reservationDTO.getStatus() : existingReservation.getReservationStatus());
        existingReservation.setPaymentStatus(reservationDTO.getPaymentStatus() != null ? reservationDTO.getPaymentStatus() : existingReservation.getPaymentStatus());

        return reservationRepository.save(existingReservation);
    }

    @Override
    public List<Reservation> listAll() {
        return reservationRepository.findAll();
    }

    @Override
    public List<Reservation> reservationsForTable(TableEntity table) {
        return reservationRepository.findAllByTableAndCheckInTime(table, LocalDateTime.now());
    }

    @Override
    public Reservation getReservationById(Long reservationId) {
        return reservationRepository.findById(reservationId).orElseThrow(InvalidReservationIdException::new);
    }

    @Override
    public boolean cancelReservation(Long reservationId) {
        Optional<Reservation> optionalReservation = reservationRepository.findById(reservationId);
        if (optionalReservation.isPresent()) {
            Reservation reservation = optionalReservation.get();
            TableEntity table = reservation.getTable();

            table.removeReservation(reservation);

            reservationHistoryService.moveReservationToHistory(reservation, "Canceled", "Canceled by user");
            reservationRepository.delete(reservation);

            return true;
        } else {
            return false;
        }
    }

    @Override
    public List<Reservation> findReservationByUser(User user) {
        LocalDateTime now = LocalDateTime.now();
        return reservationRepository.findAllByUserAndCheckInTimeAfter(user, now);
    }

    @Override
    public List<Reservation> findAllByUser(User user) {
        return reservationRepository.findAllByUser(user);
    }

    @Override
    public List<Reservation> findReservationsByUserPast(User user) {
        LocalDateTime now = LocalDateTime.now();
        return reservationRepository.findAllByUserAndCheckInTimeBefore(user, now);
    }

    @Override
    public List<Reservation> findReservationsByTableAndDateRange(TableEntity table, LocalDateTime startDateTime, LocalDateTime endDateTime) {
        return reservationRepository.findByTableAndCheckInTimeBetween(table, startDateTime, endDateTime);
    }

    @Override
    public void findReservationsToMove() {
        List<Reservation> pastReservations = reservationRepository.findAllByCheckInTimeBefore(LocalDateTime.now());
        for(Reservation reservation : pastReservations) {
            reservationHistoryService.moveReservationToHistory(reservation, "successful", "/");
            reservationRepository.delete(reservation);
        }
    }

    @Override
    public void deleteReservation(Long reservationID) {
        Reservation reservation = findById(reservationID);
        reservationRepository.delete(reservation);
    }

    @Override
    public List<Reservation> findAllByRestaurant(Restaurant restaurant) {
        return reservationRepository.findAllByRestaurantAndCheckInTimeAfter(restaurant, LocalDateTime.now());
    }
}
