import React, { useEffect, useState } from "react";
import {jwtDecode} from "jwt-decode";
import axios from "axios";

const ReservationHistory = () => {
    const [reservations, setReservations] = useState([]);
    const [filteredReservations, setFilteredReservations] = useState([]);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState(null);

    const [restaurantFilter, setRestaurantFilter] = useState("");
    const [tableFilter, setTableFilter] = useState("");
    const [partySizeFilter, setPartySizeFilter] = useState("");
    const [statusFilter, setStatusFilter] = useState("");
    const [cancellationReasonFilter, setCancellationReasonFilter] = useState("");

    const [startDate, setStartDate] = useState("");
    const [endDate, setEndDate] = useState("");

    useEffect(() => {
        const fetchReservations = async () => {
            try {
                const token = localStorage.getItem("token");
                if (!token) {
                    setError("User not authenticated.");
                    setLoading(false);
                    return;
                }

                const decodedToken = jwtDecode(token);
                const userId = decodedToken.iss;

                const response = await axios.get(
                    `http://localhost:8081/api/past-reservations/${userId}`,
                    {
                        headers: { Authorization: `Bearer ${token}` },
                    }
                );

                setReservations(response.data);
                setFilteredReservations(response.data);
            } catch (err) {
                setError("Failed to fetch reservations.");
            } finally {
                setLoading(false);
            }
        };

        fetchReservations();
    }, []);

    useEffect(() => {
        let tempReservations = reservations;

        if (restaurantFilter) {
            tempReservations = tempReservations.filter(res =>
                res.restaurant?.name.toLowerCase().includes(restaurantFilter.toLowerCase())
            );
        }
        if (tableFilter) {
            tempReservations = tempReservations.filter(res =>
                res.table?.id?.toString() === tableFilter
            );
        }
        if (partySizeFilter) {
            tempReservations = tempReservations.filter(res =>
                res.partySize?.toString() === partySizeFilter
            );
        }
        if (statusFilter) {
            tempReservations = tempReservations.filter(res =>
                res.status.toLowerCase().includes(statusFilter.toLowerCase())
            );
        }
        if (cancellationReasonFilter) {
            tempReservations = tempReservations.filter(res =>
                (res.cancellationReason || "None").toLowerCase().includes(cancellationReasonFilter.toLowerCase())
            );
        }
        if (startDate && endDate) {
            const start = new Date(startDate);
            const end = new Date(endDate);

            tempReservations = tempReservations.filter(res => {
                const reservationDate = new Date(res.reservationDateTime);
                return reservationDate >= start && reservationDate <= end;
            });
        }

        setFilteredReservations(tempReservations);
    }, [
        restaurantFilter,
        tableFilter,
        partySizeFilter,
        statusFilter,
        cancellationReasonFilter,
        startDate,
        endDate,
        reservations
    ]);

    const resetFilters = () => {
        setRestaurantFilter("");
        setTableFilter("");
        setPartySizeFilter("");
        setStatusFilter("");
        setCancellationReasonFilter("");
        setStartDate("");
        setEndDate("");
    };

    if (loading) return <div>Loading...</div>;
    if (error) return <div className="alert alert-danger">{error}</div>;

    return (
        <div className="container mt-5">
            <h3 className="mb-4 text-center">Past Reservations</h3>

            <div className="row mb-4 align-items-end">
                <div className="col-md-2">
                    <input
                        type="text"
                        className="form-control"
                        placeholder="Filter by Restaurant"
                        value={restaurantFilter}
                        onChange={(e) => setRestaurantFilter(e.target.value)}
                    />
                </div>
                <div className="col-md-2">
                    <input
                        type="number"
                        className="form-control"
                        placeholder="Filter by Table ID"
                        value={tableFilter}
                        onChange={(e) => setTableFilter(e.target.value)}
                    />
                </div>
                <div className="col-md-4 d-flex gap-2">
                    <input
                        type="date"
                        className="form-control"
                        placeholder="Start date"
                        value={startDate}
                        onChange={(e) => setStartDate(e.target.value)}
                    />
                    <input
                        type="date"
                        className="form-control"
                        placeholder="End date"
                        value={endDate}
                        onChange={(e) => setEndDate(e.target.value)}
                    />
                </div>
                <div className="col-md-2">
                    <input
                        type="number"
                        className="form-control"
                        placeholder="Filter by Party Size"
                        value={partySizeFilter}
                        onChange={(e) => setPartySizeFilter(e.target.value)}
                    />
                </div>
                <div className="col-md-2">
                    <select
                        value={statusFilter}
                        onChange={(e) => setStatusFilter(e.target.value)}
                        className="form-control"
                    >
                        <option value="">Filter by status</option>
                        <option value="successful">Successful</option>
                        <option value="canceled">Canceled</option>
                    </select>
                </div>
                <div className="col-md-2 mt-2">
                    <input
                        type="text"
                        className="form-control"
                        placeholder="Filter by Cancellation Reason"
                        value={cancellationReasonFilter}
                        onChange={(e) => setCancellationReasonFilter(e.target.value)}
                    />
                </div>
                <div className="col-md-2 mt-2">
                    <button
                        onClick={resetFilters}
                        className="btn btn-outline-secondary w-100"
                    >
                        Reset Filters
                    </button>
                </div>
            </div>

            <div className="table-responsive">
                <table className="table table-striped table-hover table-bordered align-middle">
                    <thead className="table-dark text-center">
                    <tr>
                        <th>#</th>
                        <th>Restaurant</th>
                        <th>Table</th>
                        <th>Date & Time</th>
                        <th>Party Size</th>
                        <th>Special Requests</th>
                        <th>Status</th>
                        <th>Cancellation Reason</th>
                    </tr>
                    </thead>
                    <tbody>
                    {filteredReservations.length > 0 ? (
                        filteredReservations.map((res, index) => (
                            <tr key={res.id}>
                                <td>{index + 1}</td>
                                <td>{res.restaurant?.name || "N/A"}</td>
                                <td>{res.table?.id || "N/A"}</td>
                                <td>{new Date(res.reservationDateTime).toLocaleString()}</td>
                                <td>{res.partySize}</td>
                                <td>{res.specialRequests || "None"}</td>
                                <td>{res.status}</td>
                                <td>{res.cancellationReason || "None"}</td>
                            </tr>
                        ))
                    ) : (
                        <tr>
                            <td colSpan="8" className="text-center">No reservations found.</td>
                        </tr>
                    )}
                    </tbody>
                </table>
            </div>
        </div>
    );
};

export default ReservationHistory;