import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import axios from 'axios';
import 'bootstrap/dist/css/bootstrap.min.css';
import { useParams } from 'react-router-dom';
import StarRating from "./StarRating";


const RestaurantDetails = () => {
    const navigate = useNavigate();
    const { id } = useParams();

    const [restaurant, setRestaurant] = useState(null);
    const [selectedTableId, setSelectedTableId] = useState('');
    const [selectedDate, setSelectedDate] = useState('');
    const [selectedTime, setSelectedTime] = useState('');
    const [timeOptions, setTimeOptions] = useState([]);
    const [selectedTable, setSelectedTable] = useState(null);

    useEffect(() => {
        const fetchRestaurantDetails = async () => {
            try {
                if (!id) return;
                const response = await axios.get(`http://localhost:8081/api/restaurants/${id}`);
                setRestaurant(response.data);
            } catch (error) {
                console.error('Error fetching restaurant details:', error);
            }
        };

        fetchRestaurantDetails();
    }, [id]);

    useEffect(() => {
        if (!selectedTableId) return;

        const fetchTableDetails = async () => {
            try {
                const response = await axios.get(`http://localhost:8081/api/tables/${selectedTableId}`);
                setSelectedTable(response.data);
            } catch (error) {
                console.error('Error fetching table details:', error);
            }
        };

        fetchTableDetails();
    }, [selectedTableId]);

    const today = new Date();
    const year = today.getFullYear();
    const month = String(today.getMonth() + 1).padStart(2, '0');
    const day = String(today.getDate()).padStart(2, '0');
    const formattedDate = `${year}-${month}-${day}`;
    const parseOperatingHours = (operatingHours) => {
        const [start, end] = operatingHours.split('-');
        const [startHour, startMinute] = start.split(':').map(Number);
        const [endHour, endMinute] = end.split(':').map(Number);

        const startTime = new Date();
        startTime.setHours(startHour, startMinute, 0, 0);

        const endTime = new Date();
        endTime.setHours(endHour < startHour ? endHour + 24 : endHour, endMinute, 0, 0);

        return { startTime, endTime };
    };

    const generateTimeOptions = (operatingHours) => {
        const { startTime, endTime } = parseOperatingHours(operatingHours);
        const now = new Date();

        const selectedDateObj = new Date(selectedDate);
        const isToday = selectedDateObj.toDateString() === now.toDateString();

        let currentTime = isToday ? roundToNext15Minutes(new Date()) : new Date(startTime);

        const options = [];
        while (currentTime <= endTime) {
            options.push(currentTime.toTimeString().slice(0, 5));
            currentTime = new Date(currentTime.getTime() + 15 * 60 * 1000);
        }

        return options;
    };

    useEffect(() => {
        if (restaurant && selectedDate) {
            const options = generateTimeOptions(restaurant.operatingHours);
            setTimeOptions(options);
        }
    }, [restaurant, selectedDate]);

    const handleTableSelect = (event) => {
        setSelectedTableId(event.target.value);
    };

    const handleReservationConfirmation = () => {
        const encodedTableId = encodeURIComponent(selectedTableId);
        const encodedDateTime = encodeURIComponent(`${selectedDate}T${selectedTime}`);
        const encodedRestaurantId = encodeURIComponent(restaurant.restaurantId);

        navigate(`/reservationConfirmation/${encodedTableId}/${encodedDateTime}/${encodedRestaurantId}`);
    };

    const roundToNext15Minutes = (date) => {
        const minutes = date.getMinutes();
        const remainder = minutes % 15;
        if (remainder === 0) return date;

        date.setMinutes(minutes + 15 - remainder);
        date.setSeconds(0, 0);
        return date;
    };

    return (
        <div className="container">
            {restaurant && (
                <>
                    <h2 className="card-title">
                        {restaurant.name} <StarRating key={restaurant.id} rating={restaurant.rating} />
                    </h2>
                    <div className="restaurant-details">
                        <p>Operating hours: {restaurant.operatingHours}</p>
                        <p>Cuisine: {restaurant.cuisineType}</p>
                        <p>Address: {restaurant.address}</p>
                        <p>Phone: {restaurant.phone}</p>
                        <p>
                            Website: <a href={restaurant.website}>{restaurant.website}</a>
                        </p>
                        <p>Social Media Links: {restaurant.socialMediaLinks}</p>

                        <label>Select Table:</label>
                        <select
                            className="form-select"
                            aria-label="Select Table"
                            onChange={handleTableSelect}
                            value={selectedTableId}
                        >
                            <option value="">Select Table</option>
                            {restaurant.tablesList.map((table) => (
                                <option key={table.id} value={table.id}>
                                    {`Capacity: ${table.capacity} - ${table.location}`}
                                </option>
                            ))}
                        </select>

                        {selectedTable && (
                            <>
                                <label>Select Date:</label>
                                <input
                                    type="date"
                                    className="form-control mt-2"
                                    onChange={(event) => setSelectedDate(event.target.value)}
                                    value={selectedDate}
                                    min={formattedDate}
                                    onKeyDown={(e) => e.preventDefault()}
                                />
                                {!selectedDate && <p style={{ color: "red" }}>Please select a valid date.</p>}

                                <label>Select Time:</label>
                                <select
                                    className="form-select mt-2"
                                    onChange={(event) => setSelectedTime(event.target.value)}
                                    value={selectedTime}
                                    disabled={!selectedDate}
                                >
                                    <option value="">Select Time</option>
                                    {timeOptions.map((time, index) => (
                                        <option key={index} value={time}>
                                            {time}
                                        </option>
                                    ))}
                                </select>
                            </>
                        )}

                        <br />
                        <button
                            className="btn btn-primary"
                            onClick={handleReservationConfirmation}
                            disabled={!selectedTableId || !selectedDate || !selectedTime}
                        >
                            Confirm Reservation
                        </button>
                    </div>
                </>
            )}
        </div>
    );
};

export default RestaurantDetails;