<?php

namespace App\Http\Controllers\Dashboard;

use App\Helpers\Alert;
use App\Http\Requests\Dashboard\FolderRequest;
use App\Http\Requests\Dashboard\FileNameRequest;
use App\Models\Department;
use App\Models\FileType;
use App\Models\Folder;
use App\Models\File;
use App\Models\User;
use App\Notifications\NewFolderCreated;
use Carbon\Carbon;
use Illuminate\Filesystem\Filesystem;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Facades\Storage;
use App\Http\Controllers\Controller;

class FoldersController extends Controller
{
    public function index(Request $request)
    {
        $deptName = "";
        $deptCode = "";

        if ($request->query('id')) {
            $deptName = Department::find($request->query('id'))->getOriginal('name');
            $deptCode = Department::find($request->query('id'))->getOriginal('code');
            $foldersInDeptSort = Folder::with('department')->when($request->has('id'), function ($query) use ($request) {
                $query->where('department_id', $request->query('id'));
            });

            if ($request->query('sort') == 'newest') {
                $folders = $foldersInDeptSort->orderBy('created_at', 'desc')->paginate(12);
            }
            else if ($request->query('sort') == 'name') {
                $folders = $foldersInDeptSort->orderBy('name', 'asc')->paginate(12);
            }
            else if ($request->query('sort') == 'no_of_files') {
                $folders = $foldersInDeptSort->orderBy('no_of_files', 'desc')->paginate(12);
            }
            else if($request->query('sort') == 'count'){
                $total = $foldersInDeptSort->folder->files->count();
                $folders = $foldersInDeptSort->orderBy($total, 'asc')->paginate(12);
            }
            else {
                $folders = Folder::where('department_id', $request->query('id'))->paginate(12);
            }
        } else {
            if ($request->query('sort') == 'newest') {
                $folders = Folder::orderBy('created_at', 'desc')->paginate(12);
            }
            else if ($request->query('sort') == 'name') {
                $folders = Folder::orderBy('name', 'asc')->paginate(12);
            }
            else if ($request->query('sort') == 'no_of_files') {
                $folders = Folder::orderBy('no_of_files', 'desc')->paginate(12);
            }
            else if ($request->query('sort') == 'important') {
                $folders = Folder::where('is_important', true)->paginate(12);
            }
            else if ($request->query('sort') == 'recent') {
                $folders = Folder::orderBy('created_at', 'desc')->paginate(12);
            } else if ($request->query('search')) {

                $result = collect();
                $queries = explode(" ", $request->search);
                foreach ($queries as $query) {
                    $result->push(Folder::where("arch_id", "like", "%{$query}%")->orWhere("name", "like", "%{$query}%")->get());
                }
                $result = $result->flatten();
                $folders = $result;
            } else {
                $folders = Folder::paginate(12);
            }
        }

        $departments = Department::all();

        $diskTotal = disk_total_space('/');
        $diskTotalSize = $diskTotal / 1073741824;

        $diskFree = disk_free_space('/');
        $used = $diskTotal - $diskFree;

        $diskUsedSize = $used / 1073741824;
        $diskUse1 = round(100 - (($diskUsedSize / $diskTotalSize) * 100));
        $diskUse = round(100 - ($diskUse1)) . '%';

        return view("dashboard.folders.index")->with([
            "folders" => $folders,
            "currentUser" => auth()->user(),
            "departments" => $departments,
            "docsCount" => Department::withCount('folder')->get(),
            "totalDocs" => Folder::all()->count(),
            "countImportant" => Folder::where('is_important', true)->get()->count(),
            "diskTotal" => $diskTotal,
            "diskTotalSize" => $diskTotalSize,
            "diskUse" => $diskUse,
            "diskUsedSize" => $diskUsedSize,
            "deptName" => $deptName,
            "deptCode" => $deptCode,
            "fileTypes" => '.' . implode(',.', explode(',', FileType::find('1')->mimes))
        ]);

    }

    public function create()
    {
        return view("dashboard.folders.create")->with([
            "departments" => Department::all()
        ]);
    }

    public function store(FolderRequest $request)
    {
        $folder = new Folder();
        $user = auth()->user();
        $department = Department::find($request->department);

        $folder->user()->associate($user);
        $folder->department()->associate($department);
        $folder->department()->increment('no_of_folders');

        $folder->arch_id = $request->arch_id;
        $folder->name = $request->name;
        $folder->note = $request->note;

        if($request->has('is_important')){
            $folder->is_important = true;
        }else{
            $folder->is_important = false;
        }

        $location = $folder->department->location . DIRECTORY_SEPARATOR . $request->name;

        if (!Storage::disk('uploads')->has($location)) {
            Storage::disk('uploads')->makeDirectory($location);
        }

        $users = User::all();
        Notification::send($users, new NewFolderCreated("New folder created"));

        $folder->location = $location;

        $folder->save();

        if ($request->has('file_item')) {
            foreach ($request->file_item as $file) {
                $fileName = $folder->name . '-' . uniqid() . '.' . $file->getClientOriginalExtension();
                $file->storeAs($location . DIRECTORY_SEPARATOR, $fileName);
                $newFile = new File();
                $newFile->name = $fileName;
                $newFile->location = $location . DIRECTORY_SEPARATOR . $fileName;
                $newFile->folder()->associate($folder);
                $newFile->folder()->increment('no_of_files');
                $newFile->save();
            }
        }

        Alert::flash("New folder created successfully");

        return redirect()->route("dashboard.folders.index");
    }

    public function editShow($id)
    {
        return view("dashboard.folders.edit")->with([
            "folder" => Folder::findOrFail($id),
            "departments" => Department::all(),
            "files" => File::where('folder_id', $id)->get(),
            "excelExt" => array("xls", "xlsx", "xls", "csv"),
            "textExt" => array("txt", "doc", "docx"),
            "imageExt" => array("png", "jpg", "jpeg"),
        ]);
    }

    public function edit(FolderRequest $request, $id)
    {
        $folder = Folder::findOrFail($id);
        $files = File::where('folder_id', $id)->get();

        $department = Department::find($request->department);

        $folder->department()->increment('no_of_folders');

        $oldLocation = $folder->location;

        $folder->name = $request->name;
        $folder->arch_id = $request->arch_id;
        $folder->note = $request->note;
        $folder->updated_at = Carbon::now();

       $newLocation = Department::find($request->department)->location . DIRECTORY_SEPARATOR . $request->name;

        if($folder->department_id != $request->department){
            $folder->department()->decrement('no_of_folders');
            if (!Storage::disk('uploads')->has($newLocation)) {
                Storage::disk('uploads')->move($oldLocation, $newLocation);
                foreach($files as $file) {
                    $file->location = $newLocation . DIRECTORY_SEPARATOR . $file->name;
                    $file->save();
                }
            }
        }
         if($folder->isDirty('name')) {
            if (!Storage::disk('uploads')->has($newLocation)) {
                Storage::disk('uploads')->move($oldLocation, $newLocation);
                foreach($files as $file) {
                    $file->location = $newLocation . DIRECTORY_SEPARATOR . $file->name;
                    $file->save();
                }
            }
        }

        $folder->department()->associate($department);

        $folder->location = $newLocation;

        if ($request->has('file_item')) {
            foreach ($request->file_item as $file) {
                $fileName = $folder->name . '-' . uniqid() . '.' . $file->getClientOriginalExtension();
                    $file->storeAs($newLocation . DIRECTORY_SEPARATOR, $fileName);
                    $newFile = new File();
                    $newFile->name = $fileName;
                    $newFile->location = $newLocation . DIRECTORY_SEPARATOR . $fileName;
                    $newFile->folder()->associate($folder);
                    $newFile->folder()->increment('no_of_files');
                    $newFile->save();
            }
        }

        $folder->save();

        Alert::flash("Folder edited successfully");

        return redirect()->back();
    }

    public function toggleImportant($id)
    {
        $folder = Folder::find($id);
        $folder->is_important = !$folder->is_important;
        $folder->save();

        if ($folder->is_important == true)
            Alert::flash("Folder marked as important successfully");
        else
            Alert::flash("Folder marked as not important successfully");

        return redirect()->back();
    }

    public function destroy($id)
    {
        $folder = Folder::find($id);
        $files = File::where('folder_id', $id)->get();
        if (auth()->user()->hasPermission("delete_all_folders") && !$folder->is_important) {

            foreach ($files as $file) {
                $file->delete();
            }
            $folder->delete();
            $location = $folder->department->location . DIRECTORY_SEPARATOR . $folder->name;
            Storage::disk('uploads')->deleteDirectory($location);
            $folder->department()->decrement('no_of_folders');
            Alert::flash($folder->name . " deleted successfully");
            return redirect()->back();
        }
        Alert::flash($folder->name . " is important", "error");
        return redirect()->back();
    }

    public function downloadfolder(Request $request, $id)
    {
        try {
            $folder = Folder::find($id);
            $FileSystem = new Filesystem();
            $zip_file = Storage::disk('uploads')->path('Folder.zip');
            $zip = new \ZipArchive();
            $zip->open($zip_file, \ZipArchive::CREATE | \ZipArchive::OVERWRITE);

            $path = Storage::disk('uploads')->path($folder->department->location . DIRECTORY_SEPARATOR . $folder->name);

            $files = new \RecursiveIteratorIterator(new \RecursiveDirectoryIterator($path));

            $filesInDoc = $FileSystem->allFiles($path);

            foreach ($files as $file) {
                if (!$file->isDir()) {
                    $filePath = $file->getRealPath();
                    // extracting filename with substr/strlen
                    $relativePath = substr($filePath, strlen($path) + 1);
                    $zip->addFile($filePath, $relativePath);
                }
            }
            $zip->close();
            $headers = array('Content-Type' => 'application/octet-stream',);
            $zip_new_name = Carbon::now()->format('d.m.Y - H:i') . $folder->name . '.zip';
            return response()->download($zip_file, $zip_new_name, $headers);
        }
        catch (\Exception $e) {
            //abort(403, "No files found");
            Alert::flash("This folder has no files", "warning");
            return redirect()->back();
        }
    }

    public function files(Request $request, $id)
    {
        $folder = Folder::findOrFail($id);
        $files = File::where('folder_id', $id);
        $deptId = $folder->department_id;
        $folders = Folder::where('department_id', $deptId)->get();

        $queries = explode(" ", $request->search);

        if ($request->query('search')) {
            $result = collect();

            foreach ($queries as $query) {
                $result->push($files->where("name", "like", "%{$query}%")->get());
            }
            $result = $result->flatten();
            $files = $result;
        }
        else {
            $files = File::where('folder_id', $id)->paginate(12);
        }

        return view("dashboard.folders.files")->with([
            "folder" => $folder,
            "departments" => Department::all(),
            "files" => $files,
            "excelExt" => array("xls", "xlsx", "xls", "csv"),
            "textExt" => array("txt", "doc", "docx"),
            "imageExt" => array("png", "jpg", "jpeg"),
            "folders" => $folders,
            "fileTypes" => '.' . implode(',.', explode(',', FileType::find('1')->mimes))
        ]);
    }
}

