<?php

namespace App\Http\Controllers\Dashboard;

use App\Helpers\Alert;
use App\Http\Requests\Dashboard\DocumentRequest;
use App\Models\Department;
use App\Models\Document;
use App\Models\File;
use App\Models\Files;
use App\Models\User;
use App\Notifications\NewDocumentCreated;
use App\Services\UploadService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Mews\Purifier\Facades\Purifier;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Notification;

class DocumentsController extends Controller
{
    public function index()
    {
        if (auth()->user()->hasPermission("manage_all_documents"))
            $documents = Document::all();
        else
            $documents = Document::where("user_id", auth()->user()->id)->get();

        return view("dashboard.documents.index")->with([
            "documents" => $documents,
            "currentUser" => auth()->user(),
            "departments" => Department::all(),
            "docsCount" => Department::withCount('document')->get(),
            'totalDocs' => Document::all()->count()
        ]);
    }

    public function create()
    {
        return view("dashboard.documents.create")->with([
            "departments" => Department::all()
        ]);
    }

    public function editShow($id)
    {
//        if (!auth()->user()->hasPermission("edit_all_documents")) {
//            return redirect()->route("dashboard.documents.index");
//        }

        return view("dashboard.documents.edit")->with([
            "document" => Document::findOrFail($id),
            "departments" => Department::all()
        ]);
    }

    public function edit(DocumentRequest $request, $id)
    {
        $document = Document::findOrFail($id);

        $department = Department::find($request->department);

        $document->department()->associate($department);

        $document->name = $request->name;
        $document->arch_id = $request->arch_id;
        $document->description = $request->description;

        $document->save();

        Alert::flash("Document edited successfully");

        return redirect()->route("dashboard.documents.index");
    }

    public function store(DocumentRequest $request, UploadService $uploadService)
    {
        $document = new Document();
        $user = auth()->user();
        $department = Department::find($request->department);

        $document->user()->associate($user);
        $document->department()->associate($department);

        $document->arch_id = $request->arch_id;
        $document->name = $request->name;
        $document->description = $request->description;

        if(!Storage::disk('local')->has($document->department()->pluck('location')->join("") . '/' . $request->arch_id)){
            Storage::disk('local')->makeDirectory($document->department()->pluck('location')->join("") . '/' . $request->arch_id);
        }

        $documentFile = $uploadService->upload(File::class, [
            "file_item" => $request->file_item,
        ], "link", true);

        $document->save();

        foreach ($documentFile as $df) {
            $file = File::find($df);
            $file->document()->associate($document);
            $file->save();
        }

            Alert::flash("New document created successfully");

        return redirect()->route("dashboard.documents.index");
    }

    public function getDocumentsInDepartment($id)
    {
        $documents = Department::findOrFail($id)->document()->get();
        $department = Department::findOrFail($id);

        return view('dashboard.documents.department')
            ->with([
                'documents' => $documents,
                'departments' => Department::all(),
                'totalDocs' => Document::all()->count(),
                'department' => $department

        ]);
    }

}
