<?php

namespace App\Http\Controllers\Dashboard;

use App\Helpers\Alert;
use App\Http\Controllers\Controller;
use App\Http\Requests\Dashboard\FileNameRequest;
use App\Http\Requests\Dashboard\FileRequest;
use App\Models\FileType;
use App\Models\Folder;
use App\Models\File;
use App\Models\User;
use App\Notifications\NewFileCreated;
use Carbon\Carbon;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Facades\Storage;

class FilesController extends Controller
{
    public function index ()
    {
        return view("dashboard.files.index")->with([
            "files" => File::all(),
            "folders" => Folder::all(),
            "excelExt" => array("xls", "xlsx", "xls", "csv"),
            "textExt" => array("txt", "doc", "docx"),
            "imageExt" => array("png", "jpg", "jpeg"),
            "fileTypes" => '.' . implode(',.', explode(',', explode(':', FileType::find('1')->mimes)[1]))
        ]);
    }

    public function deleteFile($id)
    {
        $file = File::find($id);
        $file->delete();
        Storage::disk('uploads')->delete($file->location);

        Alert::flash($file->name . " deleted successfully");

        return redirect()->back();
    }

    public function store(FileRequest $request)
    {
        $file = new File();

        $folder = Folder::find($request->folder);

        $location = $folder->location;

        $users = User::all();

        if ($request->has('file_item')) {
            foreach ($request->file_item as $file) {
                $fileName = $folder->name . '-' . uniqid() . '.' . $file->getClientOriginalExtension();
                $file->storeAs($location . DIRECTORY_SEPARATOR, $fileName);
                $newFile = new File();
                $newFile->name = $fileName;
                $newFile->location = $location . DIRECTORY_SEPARATOR . $fileName;
                $newFile->folder()->associate($folder);
                $newFile->save();
            }
            Notification::send($users, new NewFileCreated("New files added"));

            Alert::flash("New files added successfully");

            return redirect()->route("dashboard.files.index");
        }
        else {
            Alert::flash("No files were uploaded", "error");

            return redirect()->route("dashboard.files.index");
        }
    }

    public function downloadFile($id)
    {
        $file = File::find($id);
        return Storage::download($file->location);
    }

    public function renameFile(FileNameRequest $request, $id)
    {
        $file = File::find($id);
        $fileExtension = explode('.', $file->name)[1];

        $file->name = $request->name . '.' . $fileExtension;
        $newLocation = 'Departments' . DIRECTORY_SEPARATOR . explode(DIRECTORY_SEPARATOR, $file->location)[1] . DIRECTORY_SEPARATOR . explode(DIRECTORY_SEPARATOR, $file->location)[2] . DIRECTORY_SEPARATOR . $file->name;

        if(Storage::disk('uploads')->has($newLocation)) {
            Alert::flash("A file with the same name already exists", "error");
            return redirect()->back();
        }
        else {
            Storage::disk('uploads')->move($file->location, $newLocation);

            $file->location = $newLocation;
            $file->updated_at = Carbon::now();
            $file->save();

            Alert::flash($file->name . " updated successfully");
            return redirect()->back();
        }
    }
}
