<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Support\Str;
use Illuminate\Notifications\Notifiable;
use Illuminate\Foundation\Auth\User as Authenticatable;

class User extends Authenticatable
{
    use Notifiable;
    use HasFactory;

    protected $table = "users";

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        "name",
        "surname",
        "username",
        "password",
        "email",
        "phone_number",
        "avatar",
        "role_id"
    ];

    /**
     * The attributes that should be hidden for arrays.
     *
     * @var array
     */
    protected $hidden = [
        "password",
        "remember_token",
        "is_active"
    ];

    /**
     * The attributes that should be cast to native types.
     *
     * @var array
     */
    protected $casts = [];


    public function role() {
        return $this->belongsTo(Role::class);
    }

    public function hasPermission($permission, $id = null, $any = false) {

        $userPermissions = null;
        $flag = null;

        if($id != null) $userPermissions = User::find($id)->role->permission->pluck("name");
        else $userPermissions = $this->role->permission->pluck("name");

        if($any) {
            foreach($permission as $p) {
                if($this->hasPermission($p)) {
                    return true;
                }
            }
            return false;
        }

        if(is_string($permission)) {
            return $userPermissions->contains($permission);
        }

        if(is_array($permission)) {
            foreach($permission as $p) {
                if($this->hasPermission($p)) {
                    $flag = true;
                } else {
                    $flag = false; break;
                }
            }
        }

        return $flag;
    }

    public function hasAllPermissions(array $permissions = array(), $id = null, $any = false) {
        return $this->hasPermission($permissions, $id, $any);
    }

    public function hasAnyPermission(array $permissions = array(), $id = null, $any = true) {
        return $this->hasPermission($permissions, $id, $any);
    }

    public function hasRole($role) {
        return $role === $this->role->name;
    }

    public function isAdmin() {
        return $this->hasRole("Admin");
    }

    public function isAdminOrEditor() {
        return $this->hasRole("Admin") || $this->hasRole("Referent");
    }

    public function getFullName($id = null) {

        if($id != null) {
            $user = User::find($id);
            return $user->name . " " . $user->surname;
        }

        return $this->name . " " . $this->surname;
    }

    public function generateTemporaryPassword($length = 20) {
        return bcrypt(Str::random($length));
    }

    public function generateSecurityCode($min = 10000, $max = 99999) {
        return rand($min, $max);
    }

    public function generateVerifyToken($length = 32) {
        return Str::random($length);
    }

    public function getCreatedByName()
    {
        return User::where('id', $this->created_by)->pluck('username')->first();
    }
}
