<?php

namespace App\Http\Controllers\Dashboard;

use App\Helpers\Alert;
use App\Http\Requests\Dashboard\DocumentRequest;
use App\Models\Department;
use App\Models\Document;
use App\Models\File;
use App\Services\UploadService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use App\Http\Controllers\Controller;

class DocumentsController extends Controller
{
    public function index(Request $request)
    {
        $queries = explode(" ", $request->search);
        $result = collect();

        foreach ($queries as $query) {
            $result->push(Document::where("arch_id", "like", "%{$query}%")->orWhere("name", "like", "%{$query}%")->get());
        }

        $result = $result->flatten();


        if ($request->query('id')) {

            $documentsInDeptSort = Document::with('department')->when($request->has('id'), function($query) use ($request) {
                $query->where('department_id', $request->query('id'));
            });

            if($request->query('sort') == 'newest') {
                $documents = $documentsInDeptSort->orderBy('created_at', 'desc')->paginate(20);
            }
            else if($request->query('sort') == 'name') {
                $documents = $documentsInDeptSort->orderBy('name', 'asc')->paginate(20);
            }
            else{
                $documents = Document::where('department_id', $request->query('id'))->paginate(20);
            }
        }
        else {
            if($request->query('sort') == 'newest') {
                $documents = Document::orderBy('created_at', 'desc')->paginate(20);
            }
            else if($request->query('sort') == 'name') {
                $documents = Document::orderBy('name', 'asc')->paginate(20);
            }
            else if($request->query('sort') == 'important'){
                $documents = Document::where('is_important', true)->paginate(20);
            }
            else if($request->query('sort') == 'recent') {
               $documents = Document::orderBy('created_at', 'desc')->paginate(20);
            }
            else if($request->query('search')){
                $result = collect();

                foreach ($queries as $query) {
                    $result->push(Document::where("arch_id", "like", "%{$query}%")->orWhere("name", "like", "%{$query}%")->get());
                }
                $result = $result->flatten();
                $documents = $result;
            }
            else
            {
                $documents = Document::paginate(20);
            }
        }

        $departments = Department::all();

        $diskTotal = disk_total_space('/');
        $diskTotalSize = $diskTotal / 1073741824;

        $diskFree  = disk_free_space('/');
        $used = $diskTotal - $diskFree;

        $diskUsedSize = $used / 1073741824;
        $diskUse1 = round(100 - (($diskUsedSize / $diskTotalSize) * 100));
        $diskUse = round(100 - ($diskUse1)) . '%';

        return view("dashboard.documents.index")->with([
            "documents" => $documents,
            "currentUser" => auth()->user(),
            "departments" => $departments,
            "docsCount" => Department::withCount('document')->get(),
            "totalDocs" => Document::all()->count(),
            "countImportant" => Document::where('is_important', true)->get()->count(),
            "diskTotal" => $diskTotal,
            "diskTotalSize" => $diskTotalSize,
            "diskUse" => $diskUse,
            "diskUsedSize" => $diskUsedSize

        ]);

    }

    public function create()
    {
        return view("dashboard.documents.create")->with([
            "departments" => Department::all()
        ]);
    }

    public function store(DocumentRequest $request, UploadService $uploadService)
    {
        $document = new Document();
        $user = auth()->user();
        $department = Department::find($request->department);

        $document->user()->associate($user);
        $document->department()->associate($department);

        $document->arch_id = $request->arch_id;
        $document->name = $request->name;
        $document->description = $request->description;

        if (!Storage::disk('local')->has($document->department()->pluck('location')->join("") . '/' . $request->arch_id)) {
            Storage::disk('local')->makeDirectory($document->department()->pluck('location')->join("") . '/' . $request->arch_id);
        }

        $documentFile = $uploadService->upload(File::class, [
            "file_item" => $request->file_item,
        ], "link", true);

        $document->save();

        foreach ($documentFile as $df) {
            $file = File::find($df);
            $file->document()->associate($document);
            $file->save();
        }

        Alert::flash("New document created successfully");

        return redirect()->route("dashboard.documents.index");
    }

    public function editShow($id)
    {
//        if (!auth()->user()->hasPermission("edit_all_documents")) {
//            return redirect()->route("dashboard.documents.index");
//        }

        return view("dashboard.documents.edit")->with([
            "document" => Document::findOrFail($id),
            "departments" => Department::all()
        ]);
    }

    public function edit(DocumentRequest $request, $id)
    {
        $document = Document::findOrFail($id);

        $department = Department::find($request->department);

        $document->department()->associate($department);

        $document->name = $request->name;
        $document->arch_id = $request->arch_id;
        $document->description = $request->description;

        $document->save();

        Alert::flash("Document edited successfully");

        return redirect()->route("dashboard.documents.index");
    }

    public function toggleImportant($id)
    {
        $document = Document::find($id);
        $document->is_important = !$document->is_important;
        $document->save();

        if($document->is_important==true)
        Alert::flash("Document marked as important successfully");
        else
            Alert::flash("Document marked as not important successfully");

        return redirect()->back();
    }

    public function search(Request $request)
    {
        $queries = explode(" ", $request->q);
        $result = collect();

        foreach ($queries as $query) {
            $result->push(Document::where("arch_id", "like", "%{$query}%")->orWhere("name", "like", "%{$query}%")->get());
        }

        $result = $result->flatten();

        $departments = Department::all();

        return view("dashboard.documents.search")
            ->with("searchQuery", $request->q)
            ->with("results", $result)
            ->with("departments", $departments)
            ->with("countImportant", Document::where('is_important', true)->get()->count());
    }

    public function destroy($id)
    {
        $document = Document::find($id);
        if (auth()->user()->hasPermission("delete_all_posts")) {
            $document->delete();
            Alert::flash($document->name . " deleted successfully");
        }
        return redirect()->route("dashboard.documents.index");
    }
}
