<?php

namespace App\Http\Controllers\Dashboard;

use App\Helpers\Alert;
use App\Http\Requests\Dashboard\DocumentRequest;
use App\Http\Requests\Dashboard\FileRequest;
use App\Http\Requests\Dashboard\PasswordSettingsRequest;
use App\Models\Department;
use App\Models\Document;
use App\Models\File;
use App\Services\UploadService;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Storage;
use App\Http\Controllers\Controller;

class DocumentsController extends Controller
{
    public function index(Request $request)
    {
        $queries = explode(" ", $request->search);
        $result = collect();

        foreach ($queries as $query) {
            $result->push(Document::where("arch_id", "like", "%{$query}%")->orWhere("name", "like", "%{$query}%")->get());
        }

        $result = $result->flatten();

        $deptName = "";
        $deptCode = "";

        if ($request->query('id')) {
            $deptName = Department::find($request->query('id'))->getOriginal('name');
            $deptCode = Department::find($request->query('id'))->getOriginal('code');
            $documentsInDeptSort = Document::with('department')->when($request->has('id'), function ($query) use ($request) {
                $query->where('department_id', $request->query('id'));
            });

            if ($request->query('sort') == 'newest') {
                $documents = $documentsInDeptSort->orderBy('created_at', 'desc')->paginate(16);
            } else if ($request->query('sort') == 'name') {
                $documents = $documentsInDeptSort->orderBy('name', 'asc')->paginate(16);
            } else {
                $documents = Document::where('department_id', $request->query('id'))->paginate(16);
            }
        } else {
            if ($request->query('sort') == 'newest') {
                $documents = Document::orderBy('created_at', 'desc')->paginate(16);
            } else if ($request->query('sort') == 'name') {
                $documents = Document::orderBy('name', 'asc')->paginate(16);
            } else if ($request->query('sort') == 'important') {
                $documents = Document::where('is_important', true)->paginate(16);
            } else if ($request->query('sort') == 'recent') {
                $documents = Document::orderBy('created_at', 'desc')->paginate(16);
            } else if ($request->query('search')) {
                $result = collect();

                foreach ($queries as $query) {
                    $result->push(Document::where("arch_id", "like", "%{$query}%")->orWhere("name", "like", "%{$query}%")->get());
                }
                $result = $result->flatten();
                $documents = $result;
            } else {
                $documents = Document::paginate(20);
            }
        }

        $departments = Department::all();

        $diskTotal = disk_total_space('/');
        $diskTotalSize = $diskTotal / 1073741824;

        $diskFree = disk_free_space('/');
        $used = $diskTotal - $diskFree;

        $diskUsedSize = $used / 1073741824;
        $diskUse1 = round(100 - (($diskUsedSize / $diskTotalSize) * 100));
        $diskUse = round(100 - ($diskUse1)) . '%';

        return view("dashboard.documents.index")->with([
            "documents" => $documents,
            "currentUser" => auth()->user(),
            "departments" => $departments,
            "docsCount" => Department::withCount('document')->get(),
            "totalDocs" => Document::all()->count(),
            "countImportant" => Document::where('is_important', true)->get()->count(),
            "diskTotal" => $diskTotal,
            "diskTotalSize" => $diskTotalSize,
            "diskUse" => $diskUse,
            "diskUsedSize" => $diskUsedSize,
            "deptName" => $deptName,
            "deptCode" => $deptCode,

        ]);

    }

    public function create()
    {
        return view("dashboard.documents.create")->with([
            "departments" => Department::all()
        ]);
    }

    public function store(DocumentRequest $request)
    {
        $document = new Document();
        $user = auth()->user();
        $department = Department::find($request->department);

        $document->user()->associate($user);
        $document->department()->associate($department);

        $document->arch_id = $request->arch_id;
        $document->name = $request->name;
        $document->description = $request->description;

        $location = 'Departments' . DIRECTORY_SEPARATOR . $document->department->code . DIRECTORY_SEPARATOR . $request->name;

        if (!Storage::disk('local')->has($location)) {
            Storage::disk('local')->makeDirectory($location);
        }

        $document->save();

        if ($request->has('file_item')) {
            foreach ($request->file_item as $file) {
                $file->storeAs($location . DIRECTORY_SEPARATOR, $file->getClientOriginalName());
                $newFile = new File();
                $newFile->name = $file->getClientOriginalName();
                $newFile->location = $location . DIRECTORY_SEPARATOR . $file->getClientOriginalName();
                $newFile->document()->associate($document);
                $newFile->save();
            }
        }

        Alert::flash("New document created successfully");

        return redirect()->route("dashboard.documents.index");
    }

    public function editShow($id)
    {
        return view("dashboard.documents.edit")->with([
            "document" => Document::findOrFail($id),
            "departments" => Department::all(),
            "files" => File::where('document_id', $id)->get()
        ]);
    }

    public function edit(DocumentRequest $request, $id)
    {
        $document = Document::findOrFail($id);
        $files = File::where('document_id', $id)->get();

        $department = Department::find($request->department);

        $document->department()->associate($department);

        $oldLocation = 'Departments' . DIRECTORY_SEPARATOR . $document->department->code . DIRECTORY_SEPARATOR . $document->name;

        $document->name = $request->name;
        $document->arch_id = $request->arch_id;
        $document->description = $request->description;

        $location = 'Departments' . DIRECTORY_SEPARATOR . $document->department->code . DIRECTORY_SEPARATOR . $request->name;

        if ($document->isDirty('name')) {
            if (!Storage::disk('local')->has($location)) {
                Storage::disk('local')->move($oldLocation, $location);
                foreach($files as $file){
                    $file->location = "test";
                }
            }
        }

        if ($request->has('file_item')) {
            foreach ($request->file_item as $file) {
                $fileName = $file->getClientOriginalName();
                if (Storage::disk('local')->has($location . DIRECTORY_SEPARATOR . $fileName)) {
                    // $hasFileError = true;
                    // break;
                    $NewFileName = time() . $fileName;
                    $file->storeAs($location . DIRECTORY_SEPARATOR, $NewFileName);
                    $newFile = new File();
                    $newFile->name = rand() . $fileName;
                    $newFile->location = $location . DIRECTORY_SEPARATOR . $NewFileName;
                    $newFile->document()->associate($document);
                    $newFile->save();
                } else {
                    $file->storeAs($location . DIRECTORY_SEPARATOR, $fileName);
                    $newFile = new File();
                    $newFile->name = $fileName;
                    $newFile->location = $location . DIRECTORY_SEPARATOR . $fileName;
                    $newFile->document()->associate($document);
                    $newFile->save();
                }
            }
        }

        $document->save();

//        if($hasFileError) {
//            Alert::flash('Document with the same name exists', 'error');
//            return redirect()->route("dashboard.documents.edit", ['id' => $document->id]);
//        }



        Alert::flash("Document edited successfully");

        return redirect()->back();
    }

    public function toggleImportant($id)
    {
        $document = Document::find($id);
        $document->is_important = !$document->is_important;
        $document->save();

        if ($document->is_important == true)
            Alert::flash("Document marked as important successfully");
        else
            Alert::flash("Document marked as not important successfully");

        return redirect()->back();
    }

    public function search(Request $request)
    {
        $queries = explode(" ", $request->q);
        $result = collect();

        foreach ($queries as $query) {
            $result->push(Document::where("arch_id", "like", "%{$query}%")->orWhere("name", "like", "%{$query}%")->get());
        }

        $result = $result->flatten();

        $departments = Department::all();

        return view("dashboard.documents.search")
            ->with("searchQuery", $request->q)
            ->with("results", $result)
            ->with("departments", $departments)
            ->with("countImportant", Document::where('is_important', true)->get()->count());
    }

    public function destroy($id)
    {
        $document = Document::find($id);
        $files = File::where('document_id', $id)->get();
        if (auth()->user()->hasPermission("delete_all_documents")) {
           // @dd($files);
            foreach ($files as $file) {
                $file->delete();
            }
            $document->delete();
            $location = 'Departments' . DIRECTORY_SEPARATOR . $document->department->code . DIRECTORY_SEPARATOR . $document->name;
            Storage::disk('local')->deleteDirectory($location);
            Alert::flash($document->name . " deleted successfully");
        }
        return redirect()->route("dashboard.documents.index");
    }

    public function deleteFile($id)
    {
        $file = File::find($id);
        $file->delete();
        Storage::disk('local')->delete($file->location);

        Alert::flash($file->name . " deleted successfully");

        return redirect()->back();
    }

    public function downloadFile($id)
    {
        $file = File::find($id);
        return Storage::download($file->location);
    }

    public function renameFile(FileRequest $request, $id)
    {
        $file = File::find($id);
        $fileExtension = explode('.', $file->name)[1];

        $file->name = $request->name . '.' . $fileExtension;
        $newLocation = 'Departments' . DIRECTORY_SEPARATOR . explode(DIRECTORY_SEPARATOR, $file->location)[1] . DIRECTORY_SEPARATOR . explode(DIRECTORY_SEPARATOR, $file->location)[2] . DIRECTORY_SEPARATOR . $file->name;

        if(Storage::disk('local')->has($newLocation)) {
            Alert::flash("A file with the same name already exists", "error");
            return redirect()->back();
        }
        else {
            Storage::disk('local')->copy($file->location, $newLocation);
            Storage::disk('local')->delete($file->location);

        $file->location = $newLocation;
        $file->save();

        Alert::flash($file->name . " updated successfully");
        return redirect()->back();
        }
    }
}
