package edu.gjoko.schedlr.services;

import edu.gjoko.schedlr.entity.*;
import edu.gjoko.schedlr.repositories.*;
import lombok.AllArgsConstructor;
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.List;

import static edu.gjoko.schedlr.entity.BusinessStatus.ACTIVE;
import static edu.gjoko.schedlr.entity.BusinessStatus.NEW;

@Service
@AllArgsConstructor
public class BusinessService {

    private final BusinessRepository businessRepository;
    private final ServiceTypeRepository serviceTypeRepository;
    private final ServiceRepository serviceRepository;
    private final StakeholderService stakeholderService;

    private final StakeholderRepository stakeholderRepository;
    private BCryptPasswordEncoder bCryptPasswordEncoder;

    public void saveBusiness(Business business) {
        saveNewServiceTypes(business);
        business.getOwner().setStakeholderType(StakeholderType.BUSINESS_OWNER);
        business.getOwner().setPassword(bCryptPasswordEncoder.encode(business.getOwner().getPassword()));
        business.setBusinessStatus(NEW);
        businessRepository.save(business);
    }

    public void saveNewServiceTypes(Business business) {
        business.getServices()
                .stream()
                .forEach(service -> {
                    if (service.getServiceType().getId() == null) {
                        ServiceType serviceType = service.getServiceType();
                        serviceType.setBusinessType(business.getBusinessType());
                        service.setServiceType(serviceTypeRepository.save(serviceType));
                    }
                });
    }

    public List<Business> findAll() {
        return businessRepository.findAll();
    }

    public void updateBusinesses(List<Business> businessList) {
        businessList
                .stream()
                .forEach(business -> {
                    stakeholderService.saveOrUpdateStakeholder(business.getOwner());
                    saveOrUpdateServices(business.getServices());
                    saveOrUpdateBusiness(business);
                });
    }

    public Business findByOwner(Long ownerId) {
        var owner = new Stakeholder();
        owner.setId(ownerId);
        return businessRepository.findBusinessByOwner(owner);
    }

    public List<Business> findByBusinessTypeAndActiveStatus(Long businessTypeId) {
        return businessRepository.findBusinessesByBusinessStatusAndBusinessType_Id(ACTIVE, businessTypeId);
    }



    private void saveOrUpdateBusiness(Business business) {
        if (business.getId() != null) {
            var foundBusinessEntity = businessRepository.findById(business.getId());
            business.setCreated(foundBusinessEntity.get().getCreated());
        }
        businessRepository.save(business);
    }

    private void saveOrUpdateServices(List<edu.gjoko.schedlr.entity.Service> serviceList) {
        if (!CollectionUtils.isEmpty(serviceList)) {
            serviceList.forEach( service -> {
                if (service.getId() != null) {
                    var found = serviceRepository.findById(service.getId());
                    service.setCreated(found.get().getCreated());
                }
            });
            serviceRepository.saveAll(serviceList);
        }
    }
}
