package edu.gjoko.schedlr.services;

import edu.gjoko.schedlr.entity.Appointment;
import edu.gjoko.schedlr.entity.AppointmentStatus;
import edu.gjoko.schedlr.entity.Review;
import edu.gjoko.schedlr.repositories.AppointmentRepository;
import edu.gjoko.schedlr.repositories.ReviewRepository;
import lombok.AllArgsConstructor;
import org.springframework.stereotype.Service;

import javax.persistence.EntityNotFoundException;
import java.util.List;
import java.util.Optional;

@Service
@AllArgsConstructor
public class ReviewService {

    private final ReviewRepository reviewRepository;
    private final AppointmentRepository appointmentRepository;

    public void save(Review review, Long customerId) {
        Optional<Appointment> appointmentOptional = appointmentRepository.findById(review.getAppointment().getId());
        if (appointmentOptional.isEmpty()) {
            throw new EntityNotFoundException();
        }
        Appointment appointment = appointmentOptional.get();
        if (appointment.getCustomer().getId().equals(customerId)) {
            review.setAppointment(appointment);
            markAppointmentAsReviewed(review.getAppointment());
            calculateAndUpdateCumulatedRatingForService(appointment.getService(), review.getRating());
            reviewRepository.save(review);
        } else {
            throw new EntityNotFoundException();
        }
    }

    public List<Review> getReviewsByServiceId(Long businessId) {
        return reviewRepository.findAllByAppointment_Service_Id(businessId);
    }

    public List<Review> getReviewsByCustomerId(Long customerId) {
        return reviewRepository.findAllByAppointment_Customer_Id(customerId);
    }

    private void markAppointmentAsReviewed(Appointment appointment) {
        appointment.setAppointmentStatus(AppointmentStatus.FINISHED_AND_REVIEWED_BY_USER);
    }

    private void calculateAndUpdateCumulatedRatingForService(edu.gjoko.schedlr.entity.Service service, int rating) {
        int newReviewsCount = service.getReviewsCount() + 1;
        float currentRating = service.getRating();
        float newRating = (currentRating * service.getReviewsCount() + rating) / newReviewsCount;
        service.setReviewsCount(newReviewsCount);
        service.setRating(newRating);
    }
}
