import { Component, OnInit } from '@angular/core';
import { ShipmentService } from '../shipment.service';
import { ActivatedRoute, Router } from '@angular/router';
import { ShipmentExpense } from '../shipmentModelDefinition';
import { shipmentModelInitializer } from '../shipmentModelInitializer';
import { ExpenseTypeService } from 'src/app/ExpenseTypeEntity/expense-type.service';
import { ExpenseType } from 'src/app/ExpenseTypeEntity/expenseTypeModelDefinition';
import { reloadCurrentRoute } from 'src/assets/RouteHelper';
import { ToastrService } from 'ngx-toastr';
import { ShowErrorToaster, ShowSuccessToaster } from 'src/assets/ErrorHandler';
import { isAppInOfflineMode } from 'src/environments/app-settings';

@Component({
  selector: 'app-shipment-add-expense',
  templateUrl: './shipment-add-expense.component.html',
  styleUrls: ['./shipment-add-expense.component.scss']
})
export class ShipmentAddExpenseComponent implements OnInit {
constructor(
    private shipmentService: ShipmentService,
    private expenseTypeService: ExpenseTypeService, 
    private router: Router, 
    private route: ActivatedRoute,
    private toastr: ToastrService
  ) { }

  tableRows: string[]=["#", "ExpenseTypeName", "Amount", ""]
  disableShipmentExpense: boolean = true;
  shipmentExpenses: ShipmentExpense[] =  isAppInOfflineMode() ? [] : shipmentModelInitializer.shipmentExpenses
  expenseTypes: ExpenseType[] = [];
  shipmentExpense: ShipmentExpense = shipmentModelInitializer.createShipmentExpense
  shipmentId: number = 0

  ngOnInit(): void {
    this.route.paramMap.subscribe(params => {
      this.shipmentId = Number(params.get('shipmentId'));
      this.shipmentExpense.shipmentId = this.shipmentId
    });

    this.getExpenseTypes();
    this.getShipmentExpenses(this.shipmentId)
  }

  insertShipmentExpense(event: string): void{
    event != "" ? this.disableShipmentExpense = false : this.disableShipmentExpense = true
  }

  createShipmentExpense(): void{
    if(this.shipmentExpenses.some((x: ShipmentExpense) => x.expenseTypeId == this.shipmentExpense.expenseTypeId)){
      this.toastr.error(
        'Please remove the expense before ading another of the same type',
          'Expense Already Exists',
          {positionClass: 'toast-bottom-right'})

      return;
    }

    this.shipmentService.createShipmentExpense(this.shipmentExpense).subscribe({
      next: (createdShipmentExpense) => {
        this.shipmentExpense.expenseTypeId = 0
        this.shipmentExpense.amount = 0
        reloadCurrentRoute(this.router)
        ShowSuccessToaster(this.toastr)
      },
      error: (response) => ShowErrorToaster(this.toastr, response)
    });
  }

  getExpenseTypes(){
    this.expenseTypeService.getExpenseTypes().subscribe({
      next: (expenseTypes) => this.expenseTypes = expenseTypes,
      error: (response) => ShowErrorToaster(this.toastr, response)
    });
  }

  getShipmentExpenses(shipmentId: number){
    this.shipmentService.getShipmentExpenses(shipmentId).subscribe({
      next: (shipmentExpenses) => this.shipmentExpenses = shipmentExpenses,
      error: (response) => ShowErrorToaster(this.toastr, response)
    });
  }

  removeShipmentExpense(shipmentId: number, expenseTypeId: number){
    this.shipmentService.removeShipmentExpense(shipmentId, expenseTypeId).subscribe({
      next: (shipmentExpenses) => {
        reloadCurrentRoute(this.router)
        ShowSuccessToaster(this.toastr)
      },
      error: (response) => ShowErrorToaster(this.toastr, response)
    });
  }
}
